"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonClientProject = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const generated_python_client_source_code_1 = require("./components/generated-python-client-source-code");
const open_api_generator_ignore_file_1 = require("./components/open-api-generator-ignore-file");
/**
 * Python project containing a python client (and lambda handler wrappers) generated using OpenAPI Generator CLI
 */
class GeneratedPythonClientProject extends python_1.PythonProject {
    constructor(options) {
        super({
            sample: false,
            pytest: false,
            ...options,
        });
        /**
         * The directory in which the built layer is output
         */
        this.layerDistDir = "dist/layer";
        // Store whether we've synthesized the project
        this.synthed = false;
        // With pip and venv (default), it's useful to install our package into the shared venv to make
        // it easier for other packages in the monorepo to take dependencies on this package.
        if ((options.venv ?? true) && (options.pip ?? true)) {
            this.depsManager.installTask.exec("pip install --editable .");
        }
        // Package into a directory that can be used as a lambda layer. This is done as part of install since the end user
        // must control build order in the monorepo via explicit dependencies, and adding here means we can run as part of
        // initial project synthesis which ensures this is created regardless of whether the user has remembered to
        // configure build order.
        if (options.generateLayer) {
            const relativeLayerDir = path.join(".", this.layerDistDir, "python");
            this.depsManager.installTask.exec(`rm -rf ${relativeLayerDir}`);
            this.depsManager.installTask.exec(`pip install . --target ${relativeLayerDir}`);
        }
        // The NX monorepo will rewrite install tasks to install-py to ensure installs can be run sequentially in dependency
        // order. This runs as part of monorepo synthesis which is too late for this project since we synthesize early to
        // ensure the generated client code is available for the install phase of the api project itself. Thus, we rewrite
        // the install tasks ourselves instead.
        if (this.parent && "addImplicitDependency" in this.parent) {
            const installPyTask = this.addTask("install-py");
            this.depsManager.installTask.steps.forEach((step) => this.copyStepIntoTask(step, installPyTask));
            this.depsManager.installTask.reset();
        }
        // Use a package.json to ensure the client is discoverable by nx
        new projen_1.JsonFile(this, "package.json", {
            obj: {
                name: this.name,
                __pdk__: true,
                version: options.version,
                scripts: Object.fromEntries(this.tasks.all.map((task) => [task.name, `npx projen ${task.name}`])),
            },
            readonly: true,
        });
        new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        new generated_python_client_source_code_1.GeneratedPythonClientSourceCode(this, {
            specPath: options.specPath,
            invokeGenerator: options.generateClient,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
    /**
     * Copies the given step into the given task within this project
     * @private
     */
    copyStepIntoTask(step, task) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = this.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.GeneratedPythonClientProject = GeneratedPythonClientProject;
//# sourceMappingURL=data:application/json;base64,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