"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.concatMethodAndPath = void 0;
const constants_1 = require("./constants");
/**
 * Serialise a method and path into a single string
 */
exports.concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
// Add to methods to ensure no auth is added
const NO_AUTH_SPEC_SNIPPET = {
    security: [],
    "x-amazon-apigateway-auth": {
        type: "NONE",
    },
};
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 */
const applyMethodAuthorizer = (methodAuthorizer) => {
    if (methodAuthorizer) {
        if (methodAuthorizer.authorizerId === constants_1.DefaultAuthorizerIds.NONE) {
            return NO_AUTH_SPEC_SNIPPET;
        }
        else {
            return {
                security: [
                    {
                        [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                    },
                ],
            };
        }
    }
    return {};
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { methodAuthorizer, integration } = integrations[operationName];
    validateAuthorizerReference(methodAuthorizer, operation.security, operationName);
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": integration,
        ...applyMethodAuthorizer(methodAuthorizer),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (constants_1.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [constants_1.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
            // No auth for CORS options requests
            ...NO_AUTH_SPEC_SNIPPET,
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(constants_1.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return whether the given OpenAPI object is a reference object
 */
const isRef = (obj) => "$ref" in obj;
/**
 * Validate the construct security schemes against the security schemes in the original spec.
 * Construct-defined authorizers always override those in the spec if they have the same ID, however we validate that
 * we are not overriding an authorizer of a different type to avoid mistakes/mismatches between the spec and the
 * construct.
 * @param constructSecuritySchemes security schemes generated from the construct authorizers
 * @param existingSpecSecuritySchemes security schemes already defined in the spec
 */
const validateSecuritySchemes = (constructSecuritySchemes, existingSpecSecuritySchemes) => {
    if (existingSpecSecuritySchemes) {
        const constructSecuritySchemeIds = new Set(Object.keys(constructSecuritySchemes));
        const existingSecuritySchemeIds = new Set(Object.keys(existingSpecSecuritySchemes));
        const overlappingSecuritySchemeIds = [...constructSecuritySchemeIds].filter((id) => existingSecuritySchemeIds.has(id));
        // Any overlapping security schemes (defined in both the spec (or source smithy model) and the construct) must be of the same type.
        // The one defined in the construct will take precedence since a custom/cognito authorizer can have a resolved arn in the construct,
        // and we allow usage in the model as a forward definition with blank arn.
        overlappingSecuritySchemeIds.forEach((schemeId) => {
            if (!isRef(existingSpecSecuritySchemes[schemeId])) {
                const existingScheme = existingSpecSecuritySchemes[schemeId];
                if (constructSecuritySchemes[schemeId].type !== existingScheme.type) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructSecuritySchemes[schemeId].type} in construct but ${existingScheme.type} in OpenAPI spec or Smithy model.`);
                }
                const constructApiGatewayAuthType = constructSecuritySchemes[schemeId]["x-amazon-apigateway-authtype"];
                const existingApiGatewayAuthType = existingScheme["x-amazon-apigateway-authtype"];
                if (constructApiGatewayAuthType !== existingApiGatewayAuthType) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructApiGatewayAuthType} in construct but ${existingApiGatewayAuthType} in OpenAPI spec or Smithy model.`);
                }
            }
            else {
                throw new Error(`Security scheme with id ${schemeId} is a reference in the OpenAPI spec or Smithy model which is not supported.`);
            }
        });
    }
};
/**
 * Validate the given authorizer reference (either default or at an operation level) defined in the construct against
 * those already in the spec.
 * @param constructAuthorizer the authorizer defined in the construct
 * @param existingSpecAuthorizers the authorizers already defined in the spec
 * @param operation the operation we are validating (for clearer error messages)
 */
const validateAuthorizerReference = (constructAuthorizer, existingSpecAuthorizers, operation = "Default") => {
    // Only need to validate if defined in both - if just one we'll use that.
    if (constructAuthorizer && existingSpecAuthorizers) {
        const mergedSpecAuthorizers = Object.fromEntries(existingSpecAuthorizers.flatMap((securityRequirement) => Object.keys(securityRequirement).map((id) => [
            id,
            securityRequirement[id],
        ])));
        const specAuthorizerIds = Object.keys(mergedSpecAuthorizers);
        if (specAuthorizerIds.length > 1) {
            // Spec defined multiple authorizers but the construct can only specify one
            throw new Error(`${operation} authorizers ${specAuthorizerIds
                .sort()
                .join(", ")} defined in the OpenAPI Spec or Smithy Model would be overridden by single construct authorizer ${constructAuthorizer.authorizerId}`);
        }
        else if (specAuthorizerIds.length === 1) {
            // Single authorizer - check that they have the same id
            if (specAuthorizerIds[0] !== constructAuthorizer.authorizerId) {
                throw new Error(`${operation} authorizer ${specAuthorizerIds[0]} defined in the OpenAPI Spec or Smithy Model would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
            }
            // Check that there are no differing scopes between the construct and spec
            const specScopes = new Set(mergedSpecAuthorizers[specAuthorizerIds[0]]);
            const constructScopes = new Set(constructAuthorizer.authorizationScopes);
            const differingScopes = [
                ...[...specScopes].filter((scope) => !constructScopes.has(scope)),
                ...[...constructScopes].filter((scope) => !specScopes.has(scope)),
            ];
            if (differingScopes.length > 0) {
                throw new Error(`${operation} authorizer scopes ${[...specScopes].join(", ")} defined in the OpenAPI Spec or Smithy Model differ from those in the construct (${[
                    ...constructScopes,
                ].join(", ")})`);
            }
        }
        else if (constructAuthorizer.authorizerId !== constants_1.DefaultAuthorizerIds.NONE) {
            // "security" section of spec is [] which means no auth, but the authorizer in the construct is not the "none" authorizer.
            throw new Error(`${operation} explicitly defines no auth in the OpenAPI Spec or Smithy Model which would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
        }
    }
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
exports.prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        exports.concatMethodAndPath(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[exports.concatMethodAndPath(methodAndPath)];
    validateSecuritySchemes(options.securitySchemes, spec.components?.securitySchemes);
    validateAuthorizerReference(options.defaultAuthorizerReference, spec.security);
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, options, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: {
                // Apply any security schemes that already exist in the spec
                ...spec.components?.securitySchemes,
                // Construct security schemes override any in the spec with the same id
                ...options.securitySchemes,
            },
        },
        // Apply the default authorizer at the top level
        ...(options.defaultAuthorizerReference
            ? applyMethodAuthorizer(options.defaultAuthorizerReference)
            : {}),
    };
};
//# sourceMappingURL=data:application/json;base64,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