"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getJavaSampleSource = void 0;
exports.getJavaSampleSource = (options) => {
    const apiPackage = options.apiSrcDir.split("/").join(".");
    return {
        "ApiProps.java": `package ${apiPackage};

import software.amazon.awscdk.services.apigateway.CorsOptions;
import software.amazon.awscdk.services.apigateway.StageOptions;
import software.amazon.awscdk.services.apigateway.RestApiBaseProps;
import software.amazon.awscdk.services.apigateway.DomainNameOptions;
import software.amazon.awscdk.services.apigateway.EndpointType;
import software.amazon.awscdk.services.iam.PolicyDocument;
import software.aws.awsprototypingsdk.openapigateway.Authorizer;
import software.aws.awsprototypingsdk.openapigateway.OpenApiIntegration;

import ${options.javaClientPackage}.api.OperationConfig;

import java.util.List;
import java.util.Map;

/**
 * Properties for the Api construct
 */
@lombok.Builder @lombok.Getter
public class ApiProps implements RestApiBaseProps {
    public OperationConfig<OpenApiIntegration> integrations;
    public Authorizer defaultAuthorizer;
    public CorsOptions corsOptions;

    // Rest API Props
    public Boolean cloudWatchRole;
    public Boolean deploy;
    public StageOptions deployOptions;
    public String description;
    public Boolean disableExecuteApiEndpoint;
    public DomainNameOptions domainName;
    public String endpointExportName;
    public List<EndpointType> endpointTypes;
    public Boolean failOnWarnings;
    public Map<String, String> parameters;
    public PolicyDocument policy;
    public String restApiName;
    public Boolean retainDeployments;
}
`,
        "Api.java": `package ${apiPackage};

import com.fasterxml.jackson.databind.ObjectMapper;
import software.aws.awsprototypingsdk.openapigateway.MethodAndPath;
import software.aws.awsprototypingsdk.openapigateway.OpenApiGatewayRestApi;
import software.aws.awsprototypingsdk.openapigateway.OpenApiGatewayRestApiProps;
import software.constructs.Construct;
import ${options.javaClientPackage}.api.OperationLookup;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Type-safe construct for the API Gateway resources defined by the spec.
 * You will likely not need to modify this file, and can instead extend it and define your integrations.
 */
public class Api extends OpenApiGatewayRestApi {
    private static class SpecDetails {
        static String specPath;
        static Object spec;

        static {
            try {
                // The parsed spec is included as a packaged resource
                URL specUrl = SpecDetails.class.getClassLoader().getResource("${options.specDir}/${options.parsedSpecFileName}");

                // We'll write the parsed spec to a temporary file outside of the jar to ensure CDK can package it as an asset
                Path parsedSpecPath = Files.createTempFile("parsed-spec", ".json");
                specPath = parsedSpecPath.toString();

                ObjectMapper json = new ObjectMapper();
                spec = json.readValue(specUrl, Object.class);
                json.writeValue(parsedSpecPath.toFile(), spec);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public Api(Construct scope, String id, ApiProps props) {
        super(scope, id, OpenApiGatewayRestApiProps.builder()
                .defaultAuthorizer(props.getDefaultAuthorizer())
                .corsOptions(props.getCorsOptions())
                .operationLookup(OperationLookup.getOperationLookup()
                        .entrySet()
                        .stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, e -> MethodAndPath.builder()
                                .method(e.getValue().get("method"))
                                .path(e.getValue().get("path"))
                                .build())))
                .spec(SpecDetails.spec)
                .specPath(SpecDetails.specPath)
                .integrations(props.getIntegrations().asMap())
                // Rest API Base Props
                .cloudWatchRole(props.getCloudWatchRole())
                .deploy(props.getDeploy())
                .deployOptions(props.getDeployOptions())
                .description(props.getDescription())
                .disableExecuteApiEndpoint(props.getDisableExecuteApiEndpoint())
                .domainName(props.getDomainName())
                .endpointExportName(props.getEndpointExportName())
                .endpointTypes(props.getEndpointTypes())
                .failOnWarnings(props.getFailOnWarnings())
                .parameters(props.getParameters())
                .policy(props.getPolicy())
                .restApiName(props.getRestApiName())
                .retainDeployments(props.getRetainDeployments())
                .build());
    }
}
`,
        ...(options.sampleCode !== false
            ? {
                "SayHelloHandler.java": `package ${apiPackage};

import ${options.javaClientPackage}.api.Handlers.SayHello;
import ${options.javaClientPackage}.api.Handlers.SayHello200Response;
import ${options.javaClientPackage}.api.Handlers.SayHelloRequestInput;
import ${options.javaClientPackage}.api.Handlers.SayHelloResponse;
import ${options.javaClientPackage}.model.SayHelloResponseContent;

/**
 * An example lambda handler which uses the generated handler wrapper class (Handlers.SayHello) to manage marshalling
 * inputs and outputs.
 */
public class SayHelloHandler extends SayHello {
    @Override
    public SayHelloResponse handle(SayHelloRequestInput sayHelloRequestInput) {
        return SayHello200Response.of(SayHelloResponseContent.builder()
                .message(String.format("Hello %s", sayHelloRequestInput.getInput().getRequestParameters().getName()))
                .build());
    }
}
`,
                "SampleApi.java": `package ${apiPackage};

import software.amazon.awscdk.Duration;
import software.amazon.awscdk.services.apigateway.CorsOptions;
import software.amazon.awscdk.services.lambda.Code;
import software.amazon.awscdk.services.lambda.Function;
import software.amazon.awscdk.services.lambda.FunctionProps;
import software.amazon.awscdk.services.lambda.Runtime;
import software.aws.awsprototypingsdk.openapigateway.Authorizers;
import software.aws.awsprototypingsdk.openapigateway.Integrations;
import software.aws.awsprototypingsdk.openapigateway.OpenApiIntegration;
import ${options.javaClientPackage}.api.OperationConfig;
import software.constructs.Construct;

import java.net.URISyntaxException;
import java.util.Arrays;

/**
 * An example of how to wire lambda handler functions to API operations
 */
public class SampleApi extends Api {
    public SampleApi(Construct scope, String id) {
        super(scope, id, ApiProps.builder()
                .defaultAuthorizer(Authorizers.iam())
                .corsOptions(CorsOptions.builder()
                        .allowOrigins(Arrays.asList("*"))
                        .allowMethods(Arrays.asList("*"))
                        .build())
                .integrations(OperationConfig.<OpenApiIntegration>builder()
                        .sayHello(OpenApiIntegration.builder()
                                .integration(Integrations.lambda(
                                        new Function(scope, "say-hello", FunctionProps.builder()
                                                // Use the entire project jar for the lambda code in order to provide a simple,
                                                // "one-click" way to build the api. However this jar is much larger than necessary
                                                // since it includes cdk infrastructure, dependencies etc.
                                                // It is recommended to follow the instructions in the "Java API Lambda Handlers"
                                                // section of the open-api-gateway README to define your lambda handlers as a
                                                // separate project.
                                                .code(Code.fromAsset(SampleApi.getJarPath()))
                                                .handler("${apiPackage}.SayHelloHandler")
                                                .runtime(Runtime.JAVA_11)
                                                .timeout(Duration.seconds(30))
                                                .build())))
                                .build())
                        .build())
                .build());
    }

    private static String getJarPath() {
        try {
            // Retrieve the path of the jar in which this class resides
            return SampleApi.class.getProtectionDomain().getCodeSource().getLocation().toURI().getPath();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }
}

`,
            }
            : {}),
    };
};
//# sourceMappingURL=data:application/json;base64,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