"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const bundling_1 = require("../lib/bundling");
jest.mock('../lib/bundling', () => {
    return {
        bundle: jest.fn().mockReturnValue({
            bind: () => {
                return { inlineCode: 'code' };
            },
            bindToResource: () => { return; },
        }),
    };
});
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
    jest.clearAllMocks();
});
test('PythonFunction with defaults', () => {
    new lib_1.PythonFunction(stack, 'handler', {
        entry: 'test/lambda-handler',
    });
    expect(bundling_1.bundle).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringMatching(/@aws-cdk\/aws-lambda-python\/test\/lambda-handler$/),
    }));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
    });
});
test('PythonFunction with index in a subdirectory', () => {
    new lib_1.PythonFunction(stack, 'handler', {
        entry: 'test/lambda-handler-sub',
        index: 'inner/custom_index.py',
        handler: 'custom_handler',
    });
    expect(bundling_1.bundle).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringMatching(/@aws-cdk\/aws-lambda-python\/test\/lambda-handler-sub$/),
    }));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'inner/custom_index.custom_handler',
    });
});
test('throws when index is not py', () => {
    expect(() => new lib_1.PythonFunction(stack, 'Fn', {
        entry: 'test/lambda-handler',
        index: 'index.js',
    })).toThrow(/Only Python \(\.py\) index files are supported/);
});
test('throws when entry does not exist', () => {
    expect(() => new lib_1.PythonFunction(stack, 'Fn', {
        entry: 'notfound',
    })).toThrow(/Cannot find index file at/);
});
test('throws with the wrong runtime family', () => {
    expect(() => new lib_1.PythonFunction(stack, 'handler1', {
        entry: 'test/lambda-handler',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
    })).toThrow(/Only `PYTHON` runtimes are supported/);
});
//# sourceMappingURL=data:application/json;base64,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