"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = exports.DeploymentStatus = void 0;
const client_codedeploy_1 = require("@aws-sdk/client-codedeploy");
const logger_1 = require("./logger");
var DeploymentStatus;
(function (DeploymentStatus) {
    DeploymentStatus["CREATED"] = "Created";
    DeploymentStatus["QUEUED"] = "Queued";
    DeploymentStatus["IN_PROGRESS"] = "InProgress";
    DeploymentStatus["BAKING"] = "Baking";
    DeploymentStatus["SUCCEEDED"] = "Succeeded";
    DeploymentStatus["FAILED"] = "Failed";
    DeploymentStatus["STOPPED"] = "Stopped";
    DeploymentStatus["READY"] = "Ready";
})(DeploymentStatus = exports.DeploymentStatus || (exports.DeploymentStatus = {}));
/**
 * The lambda function called from CloudFormation for this custom resource.
 *
 * @param event
 * @returns whether the deployment is complete
 */
async function handler(event) {
    const logger = new logger_1.Logger();
    const codedeployClient = new client_codedeploy_1.CodeDeploy({});
    try {
        const resp = await codedeployClient.getDeployment({ deploymentId: event.PhysicalResourceId });
        let rollbackResp = {};
        if (resp.deploymentInfo?.rollbackInfo?.rollbackDeploymentId) {
            rollbackResp = await codedeployClient.getDeployment({ deploymentId: resp.deploymentInfo?.rollbackInfo?.rollbackDeploymentId });
        }
        logger.appendKeys({
            stackEvent: event.RequestType,
            deploymentId: event.PhysicalResourceId,
            deploymentStatus: resp.deploymentInfo?.status,
            rollbackStatus: rollbackResp?.deploymentInfo?.status,
        });
        logger.info('Checking deployment');
        // check if deployment id is complete
        switch (event.RequestType) {
            case 'Create':
            case 'Update':
                switch (resp.deploymentInfo?.status) {
                    case DeploymentStatus.SUCCEEDED:
                        logger.info('Deployment finished successfully', { complete: true });
                        return { IsComplete: true };
                    case DeploymentStatus.FAILED:
                    case DeploymentStatus.STOPPED:
                        if (rollbackResp.deploymentInfo?.status) {
                            if (rollbackResp.deploymentInfo?.status == DeploymentStatus.SUCCEEDED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.FAILED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.STOPPED) {
                                const errInfo = resp.deploymentInfo.errorInformation;
                                const error = new Error(`Deployment ${resp.deploymentInfo.status}: [${errInfo?.code}] ${errInfo?.message}`);
                                logger.error('Deployment failed', { complete: true, error });
                                throw error;
                            }
                            logger.info('Waiting for final status from a rollback', { complete: false });
                            return { IsComplete: false }; // waiting for final status from rollback
                        }
                        else {
                            const errInfo = resp.deploymentInfo.errorInformation;
                            const error = new Error(`Deployment ${resp.deploymentInfo.status}: [${errInfo?.code}] ${errInfo?.message}`);
                            logger.error('No rollback to wait for', { complete: true, error });
                            throw error;
                        }
                    default:
                        logger.info('Waiting for final status from deployment', { complete: false });
                        return { IsComplete: false };
                }
            case 'Delete':
                switch (resp.deploymentInfo?.status) {
                    case DeploymentStatus.SUCCEEDED:
                        logger.info('Deployment finished successfully - nothing to delete', { complete: true });
                        return { IsComplete: true };
                    case DeploymentStatus.FAILED:
                    case DeploymentStatus.STOPPED:
                        if (rollbackResp.deploymentInfo?.status) {
                            if (rollbackResp.deploymentInfo?.status == DeploymentStatus.SUCCEEDED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.FAILED ||
                                rollbackResp.deploymentInfo?.status == DeploymentStatus.STOPPED) {
                                logger.info('Rollback in final status', { complete: true });
                                return { IsComplete: true }; // rollback finished, we're deleted
                            }
                            logger.info('Waiting for final status from a rollback', { complete: false });
                            return { IsComplete: false }; // waiting for rollback
                        }
                        logger.info('No rollback to wait for', { complete: true });
                        return { IsComplete: true };
                    default:
                        logger.info('Waiting for final status from deployment', { complete: false });
                        return { IsComplete: false };
                }
            default:
                logger.error('Unknown request type');
                throw new Error(`Unknown request type: ${event.RequestType}`);
        }
    }
    catch (e) {
        logger.error('Unable to determine deployment status', { error: e });
        if (event.RequestType === 'Delete') {
            logger.warn('Ignoring error - nothing to do', { complete: true });
            return { IsComplete: true };
        }
        throw e;
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaXMtY29tcGxldGUubGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2Vjcy1kZXBsb3ltZW50LXByb3ZpZGVyL2lzLWNvbXBsZXRlLmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxrRUFBNkU7QUFDN0UscUNBQWtDO0FBRWxDLElBQVksZ0JBU1g7QUFURCxXQUFZLGdCQUFnQjtJQUMxQix1Q0FBbUIsQ0FBQTtJQUNuQixxQ0FBaUIsQ0FBQTtJQUNqQiw4Q0FBMEIsQ0FBQTtJQUMxQixxQ0FBaUIsQ0FBQTtJQUNqQiwyQ0FBdUIsQ0FBQTtJQUN2QixxQ0FBaUIsQ0FBQTtJQUNqQix1Q0FBbUIsQ0FBQTtJQUNuQixtQ0FBZSxDQUFBO0FBQ2pCLENBQUMsRUFUVyxnQkFBZ0IsR0FBaEIsd0JBQWdCLEtBQWhCLHdCQUFnQixRQVMzQjtBQTJCRDs7Ozs7R0FLRztBQUNJLEtBQUssVUFBVSxPQUFPLENBQUMsS0FBd0I7SUFDcEQsTUFBTSxNQUFNLEdBQUcsSUFBSSxlQUFNLEVBQUUsQ0FBQztJQUM1QixNQUFNLGdCQUFnQixHQUFHLElBQUksOEJBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUM1QyxJQUFJO1FBQ0YsTUFBTSxJQUFJLEdBQUcsTUFBTSxnQkFBZ0IsQ0FBQyxhQUFhLENBQUMsRUFBRSxZQUFZLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQztRQUM5RixJQUFJLFlBQVksR0FBd0IsRUFBRSxDQUFDO1FBQzNDLElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRSxZQUFZLEVBQUUsb0JBQW9CLEVBQUU7WUFDM0QsWUFBWSxHQUFHLE1BQU0sZ0JBQWdCLENBQUMsYUFBYSxDQUFDLEVBQUUsWUFBWSxFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsWUFBWSxFQUFFLG9CQUFvQixFQUFFLENBQUMsQ0FBQztTQUNoSTtRQUNELE1BQU0sQ0FBQyxVQUFVLENBQUM7WUFDaEIsVUFBVSxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzdCLFlBQVksRUFBRSxLQUFLLENBQUMsa0JBQWtCO1lBQ3RDLGdCQUFnQixFQUFFLElBQUksQ0FBQyxjQUFjLEVBQUUsTUFBTTtZQUM3QyxjQUFjLEVBQUUsWUFBWSxFQUFFLGNBQWMsRUFBRSxNQUFNO1NBQ3JELENBQUMsQ0FBQztRQUNILE1BQU0sQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUVuQyxxQ0FBcUM7UUFDckMsUUFBUSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ3pCLEtBQUssUUFBUSxDQUFDO1lBQ2QsS0FBSyxRQUFRO2dCQUNYLFFBQVEsSUFBSSxDQUFDLGNBQWMsRUFBRSxNQUFNLEVBQUU7b0JBQ25DLEtBQUssZ0JBQWdCLENBQUMsU0FBUzt3QkFDN0IsTUFBTSxDQUFDLElBQUksQ0FBQyxrQ0FBa0MsRUFBRSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO3dCQUVwRSxPQUFPLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDO29CQUM5QixLQUFLLGdCQUFnQixDQUFDLE1BQU0sQ0FBQztvQkFDN0IsS0FBSyxnQkFBZ0IsQ0FBQyxPQUFPO3dCQUMzQixJQUFJLFlBQVksQ0FBQyxjQUFjLEVBQUUsTUFBTSxFQUFFOzRCQUN2QyxJQUFJLFlBQVksQ0FBQyxjQUFjLEVBQUUsTUFBTSxJQUFJLGdCQUFnQixDQUFDLFNBQVM7Z0NBQ25FLFlBQVksQ0FBQyxjQUFjLEVBQUUsTUFBTSxJQUFJLGdCQUFnQixDQUFDLE1BQU07Z0NBQzlELFlBQVksQ0FBQyxjQUFjLEVBQUUsTUFBTSxJQUFJLGdCQUFnQixDQUFDLE9BQU8sRUFBRTtnQ0FDakUsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQztnQ0FDckQsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLENBQUMsY0FBYyxJQUFJLENBQUMsY0FBYyxDQUFDLE1BQU0sTUFBTSxPQUFPLEVBQUUsSUFBSSxLQUFLLE9BQU8sRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO2dDQUM1RyxNQUFNLENBQUMsS0FBSyxDQUFDLG1CQUFtQixFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO2dDQUM3RCxNQUFNLEtBQUssQ0FBQzs2QkFDYjs0QkFDRCxNQUFNLENBQUMsSUFBSSxDQUFDLDBDQUEwQyxFQUFFLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7NEJBRTdFLE9BQU8sRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyx5Q0FBeUM7eUJBQ3hFOzZCQUFNOzRCQUNMLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLENBQUM7NEJBQ3JELE1BQU0sS0FBSyxHQUFHLElBQUksS0FBSyxDQUFDLGNBQWMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLE1BQU0sT0FBTyxFQUFFLElBQUksS0FBSyxPQUFPLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQzs0QkFDNUcsTUFBTSxDQUFDLEtBQUssQ0FBQyx5QkFBeUIsRUFBRSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQzs0QkFDbkUsTUFBTSxLQUFLLENBQUM7eUJBQ2I7b0JBQ0g7d0JBQ0UsTUFBTSxDQUFDLElBQUksQ0FBQywwQ0FBMEMsRUFBRSxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO3dCQUU3RSxPQUFPLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxDQUFDO2lCQUNoQztZQUNILEtBQUssUUFBUTtnQkFDWCxRQUFRLElBQUksQ0FBQyxjQUFjLEVBQUUsTUFBTSxFQUFFO29CQUNuQyxLQUFLLGdCQUFnQixDQUFDLFNBQVM7d0JBQzdCLE1BQU0sQ0FBQyxJQUFJLENBQUMsc0RBQXNELEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQzt3QkFFeEYsT0FBTyxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsQ0FBQztvQkFDOUIsS0FBSyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUM7b0JBQzdCLEtBQUssZ0JBQWdCLENBQUMsT0FBTzt3QkFDM0IsSUFBSSxZQUFZLENBQUMsY0FBYyxFQUFFLE1BQU0sRUFBRTs0QkFDdkMsSUFBSSxZQUFZLENBQUMsY0FBYyxFQUFFLE1BQU0sSUFBSSxnQkFBZ0IsQ0FBQyxTQUFTO2dDQUNuRSxZQUFZLENBQUMsY0FBYyxFQUFFLE1BQU0sSUFBSSxnQkFBZ0IsQ0FBQyxNQUFNO2dDQUM5RCxZQUFZLENBQUMsY0FBYyxFQUFFLE1BQU0sSUFBSSxnQkFBZ0IsQ0FBQyxPQUFPLEVBQUU7Z0NBQ2pFLE1BQU0sQ0FBQyxJQUFJLENBQUMsMEJBQTBCLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztnQ0FFNUQsT0FBTyxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLG1DQUFtQzs2QkFDakU7NEJBQ0QsTUFBTSxDQUFDLElBQUksQ0FBQywwQ0FBMEMsRUFBRSxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDOzRCQUU3RSxPQUFPLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUMsdUJBQXVCO3lCQUN0RDt3QkFDRCxNQUFNLENBQUMsSUFBSSxDQUFDLHlCQUF5QixFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7d0JBRTNELE9BQU8sRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUM7b0JBQzlCO3dCQUNFLE1BQU0sQ0FBQyxJQUFJLENBQUMsMENBQTBDLEVBQUUsRUFBRSxRQUFRLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQzt3QkFFN0UsT0FBTyxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsQ0FBQztpQkFDaEM7WUFDSDtnQkFDRSxNQUFNLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLENBQUM7Z0JBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLEtBQUssQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQ2pFO0tBQ0Y7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLE1BQU0sQ0FBQyxLQUFLLENBQUMsdUNBQXVDLEVBQUUsRUFBRSxLQUFLLEVBQUUsQ0FBVSxFQUFFLENBQUMsQ0FBQztRQUM3RSxJQUFJLEtBQUssQ0FBQyxXQUFXLEtBQUssUUFBUSxFQUFFO1lBQ2xDLE1BQU0sQ0FBQyxJQUFJLENBQUMsZ0NBQWdDLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztZQUVsRSxPQUFPLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDO1NBQzdCO1FBQ0QsTUFBTSxDQUFDLENBQUM7S0FDVDtBQUNILENBQUM7QUE1RkQsMEJBNEZDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29kZURlcGxveSwgR2V0RGVwbG95bWVudE91dHB1dCB9IGZyb20gJ0Bhd3Mtc2RrL2NsaWVudC1jb2RlZGVwbG95JztcbmltcG9ydCB7IExvZ2dlciB9IGZyb20gJy4vbG9nZ2VyJztcblxuZXhwb3J0IGVudW0gRGVwbG95bWVudFN0YXR1cyB7XG4gIENSRUFURUQgPSAnQ3JlYXRlZCcsXG4gIFFVRVVFRCA9ICdRdWV1ZWQnLFxuICBJTl9QUk9HUkVTUyA9ICdJblByb2dyZXNzJyxcbiAgQkFLSU5HID0gJ0Jha2luZycsXG4gIFNVQ0NFRURFRCA9ICdTdWNjZWVkZWQnLFxuICBGQUlMRUQgPSAnRmFpbGVkJyxcbiAgU1RPUFBFRCA9ICdTdG9wcGVkJyxcbiAgUkVBRFkgPSAnUmVhZHknLFxufVxuXG4vKipcbiAqIFRoZSByZXF1ZXN0IG9iamVjdCB0aGF0IHRoZSBjdXN0b20gcmVzb3VyY2UgbGFtYmEgZnVuY3Rpb24gcmVjZWl2ZXMgZnJvbSBDbG91ZEZvcm1hdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJc0NvbXBsZXRlUmVxdWVzdCB7XG4gIC8qKlxuICAgKiBUaGUgdHlwZSBvZiBDbG91ZEZvcm1hdGlvbiByZXF1ZXN0IChlLmcuICdDcmVhdGUnLCAnVXBkYXRlJywgb3IgJ0RlbGV0ZScpXG4gICAqL1xuICBSZXF1ZXN0VHlwZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcGh5c2ljYWwgcmVzb3VyY2UgaWQuXG4gICAqL1xuICBQaHlzaWNhbFJlc291cmNlSWQ6IHN0cmluZztcbn1cblxuLyoqXG4gKiBUaGUgcmVzcG9uc2Ugb2JqZWN0IHRoYXQgdGhlIGN1c3RvbSByZXNvdXJjZSBsYW1iZGEgZnVuY3Rpb24gcmV0dXJucyB0byBDbG91ZEZvcm1hdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJc0NvbXBsZXRlUmVzcG9uc2Uge1xuICAvKipcbiAgICogVHJ1ZSBpZiBhbmQgb25seSBpZiB0aGUgZGVwbG95bWVudCBpcyBpbiBhIGZpbmFsIHN0YXRlLlxuICAgKi9cbiAgSXNDb21wbGV0ZTogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBUaGUgbGFtYmRhIGZ1bmN0aW9uIGNhbGxlZCBmcm9tIENsb3VkRm9ybWF0aW9uIGZvciB0aGlzIGN1c3RvbSByZXNvdXJjZS5cbiAqXG4gKiBAcGFyYW0gZXZlbnRcbiAqIEByZXR1cm5zIHdoZXRoZXIgdGhlIGRlcGxveW1lbnQgaXMgY29tcGxldGVcbiAqL1xuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGhhbmRsZXIoZXZlbnQ6IElzQ29tcGxldGVSZXF1ZXN0KTogUHJvbWlzZTxJc0NvbXBsZXRlUmVzcG9uc2U+IHtcbiAgY29uc3QgbG9nZ2VyID0gbmV3IExvZ2dlcigpO1xuICBjb25zdCBjb2RlZGVwbG95Q2xpZW50ID0gbmV3IENvZGVEZXBsb3koe30pO1xuICB0cnkge1xuICAgIGNvbnN0IHJlc3AgPSBhd2FpdCBjb2RlZGVwbG95Q2xpZW50LmdldERlcGxveW1lbnQoeyBkZXBsb3ltZW50SWQ6IGV2ZW50LlBoeXNpY2FsUmVzb3VyY2VJZCB9KTtcbiAgICBsZXQgcm9sbGJhY2tSZXNwOiBHZXREZXBsb3ltZW50T3V0cHV0ID0ge307XG4gICAgaWYgKHJlc3AuZGVwbG95bWVudEluZm8/LnJvbGxiYWNrSW5mbz8ucm9sbGJhY2tEZXBsb3ltZW50SWQpIHtcbiAgICAgIHJvbGxiYWNrUmVzcCA9IGF3YWl0IGNvZGVkZXBsb3lDbGllbnQuZ2V0RGVwbG95bWVudCh7IGRlcGxveW1lbnRJZDogcmVzcC5kZXBsb3ltZW50SW5mbz8ucm9sbGJhY2tJbmZvPy5yb2xsYmFja0RlcGxveW1lbnRJZCB9KTtcbiAgICB9XG4gICAgbG9nZ2VyLmFwcGVuZEtleXMoe1xuICAgICAgc3RhY2tFdmVudDogZXZlbnQuUmVxdWVzdFR5cGUsXG4gICAgICBkZXBsb3ltZW50SWQ6IGV2ZW50LlBoeXNpY2FsUmVzb3VyY2VJZCxcbiAgICAgIGRlcGxveW1lbnRTdGF0dXM6IHJlc3AuZGVwbG95bWVudEluZm8/LnN0YXR1cyxcbiAgICAgIHJvbGxiYWNrU3RhdHVzOiByb2xsYmFja1Jlc3A/LmRlcGxveW1lbnRJbmZvPy5zdGF0dXMsXG4gICAgfSk7XG4gICAgbG9nZ2VyLmluZm8oJ0NoZWNraW5nIGRlcGxveW1lbnQnKTtcblxuICAgIC8vIGNoZWNrIGlmIGRlcGxveW1lbnQgaWQgaXMgY29tcGxldGVcbiAgICBzd2l0Y2ggKGV2ZW50LlJlcXVlc3RUeXBlKSB7XG4gICAgICBjYXNlICdDcmVhdGUnOlxuICAgICAgY2FzZSAnVXBkYXRlJzpcbiAgICAgICAgc3dpdGNoIChyZXNwLmRlcGxveW1lbnRJbmZvPy5zdGF0dXMpIHtcbiAgICAgICAgICBjYXNlIERlcGxveW1lbnRTdGF0dXMuU1VDQ0VFREVEOlxuICAgICAgICAgICAgbG9nZ2VyLmluZm8oJ0RlcGxveW1lbnQgZmluaXNoZWQgc3VjY2Vzc2Z1bGx5JywgeyBjb21wbGV0ZTogdHJ1ZSB9KTtcblxuICAgICAgICAgICAgcmV0dXJuIHsgSXNDb21wbGV0ZTogdHJ1ZSB9O1xuICAgICAgICAgIGNhc2UgRGVwbG95bWVudFN0YXR1cy5GQUlMRUQ6XG4gICAgICAgICAgY2FzZSBEZXBsb3ltZW50U3RhdHVzLlNUT1BQRUQ6XG4gICAgICAgICAgICBpZiAocm9sbGJhY2tSZXNwLmRlcGxveW1lbnRJbmZvPy5zdGF0dXMpIHtcbiAgICAgICAgICAgICAgaWYgKHJvbGxiYWNrUmVzcC5kZXBsb3ltZW50SW5mbz8uc3RhdHVzID09IERlcGxveW1lbnRTdGF0dXMuU1VDQ0VFREVEIHx8XG4gICAgICAgICAgICAgICAgcm9sbGJhY2tSZXNwLmRlcGxveW1lbnRJbmZvPy5zdGF0dXMgPT0gRGVwbG95bWVudFN0YXR1cy5GQUlMRUQgfHxcbiAgICAgICAgICAgICAgICByb2xsYmFja1Jlc3AuZGVwbG95bWVudEluZm8/LnN0YXR1cyA9PSBEZXBsb3ltZW50U3RhdHVzLlNUT1BQRUQpIHtcbiAgICAgICAgICAgICAgICBjb25zdCBlcnJJbmZvID0gcmVzcC5kZXBsb3ltZW50SW5mby5lcnJvckluZm9ybWF0aW9uO1xuICAgICAgICAgICAgICAgIGNvbnN0IGVycm9yID0gbmV3IEVycm9yKGBEZXBsb3ltZW50ICR7cmVzcC5kZXBsb3ltZW50SW5mby5zdGF0dXN9OiBbJHtlcnJJbmZvPy5jb2RlfV0gJHtlcnJJbmZvPy5tZXNzYWdlfWApO1xuICAgICAgICAgICAgICAgIGxvZ2dlci5lcnJvcignRGVwbG95bWVudCBmYWlsZWQnLCB7IGNvbXBsZXRlOiB0cnVlLCBlcnJvciB9KTtcbiAgICAgICAgICAgICAgICB0aHJvdyBlcnJvcjtcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICBsb2dnZXIuaW5mbygnV2FpdGluZyBmb3IgZmluYWwgc3RhdHVzIGZyb20gYSByb2xsYmFjaycsIHsgY29tcGxldGU6IGZhbHNlIH0pO1xuXG4gICAgICAgICAgICAgIHJldHVybiB7IElzQ29tcGxldGU6IGZhbHNlIH07IC8vIHdhaXRpbmcgZm9yIGZpbmFsIHN0YXR1cyBmcm9tIHJvbGxiYWNrXG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICBjb25zdCBlcnJJbmZvID0gcmVzcC5kZXBsb3ltZW50SW5mby5lcnJvckluZm9ybWF0aW9uO1xuICAgICAgICAgICAgICBjb25zdCBlcnJvciA9IG5ldyBFcnJvcihgRGVwbG95bWVudCAke3Jlc3AuZGVwbG95bWVudEluZm8uc3RhdHVzfTogWyR7ZXJySW5mbz8uY29kZX1dICR7ZXJySW5mbz8ubWVzc2FnZX1gKTtcbiAgICAgICAgICAgICAgbG9nZ2VyLmVycm9yKCdObyByb2xsYmFjayB0byB3YWl0IGZvcicsIHsgY29tcGxldGU6IHRydWUsIGVycm9yIH0pO1xuICAgICAgICAgICAgICB0aHJvdyBlcnJvcjtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgbG9nZ2VyLmluZm8oJ1dhaXRpbmcgZm9yIGZpbmFsIHN0YXR1cyBmcm9tIGRlcGxveW1lbnQnLCB7IGNvbXBsZXRlOiBmYWxzZSB9KTtcblxuICAgICAgICAgICAgcmV0dXJuIHsgSXNDb21wbGV0ZTogZmFsc2UgfTtcbiAgICAgICAgfVxuICAgICAgY2FzZSAnRGVsZXRlJzpcbiAgICAgICAgc3dpdGNoIChyZXNwLmRlcGxveW1lbnRJbmZvPy5zdGF0dXMpIHtcbiAgICAgICAgICBjYXNlIERlcGxveW1lbnRTdGF0dXMuU1VDQ0VFREVEOlxuICAgICAgICAgICAgbG9nZ2VyLmluZm8oJ0RlcGxveW1lbnQgZmluaXNoZWQgc3VjY2Vzc2Z1bGx5IC0gbm90aGluZyB0byBkZWxldGUnLCB7IGNvbXBsZXRlOiB0cnVlIH0pO1xuXG4gICAgICAgICAgICByZXR1cm4geyBJc0NvbXBsZXRlOiB0cnVlIH07XG4gICAgICAgICAgY2FzZSBEZXBsb3ltZW50U3RhdHVzLkZBSUxFRDpcbiAgICAgICAgICBjYXNlIERlcGxveW1lbnRTdGF0dXMuU1RPUFBFRDpcbiAgICAgICAgICAgIGlmIChyb2xsYmFja1Jlc3AuZGVwbG95bWVudEluZm8/LnN0YXR1cykge1xuICAgICAgICAgICAgICBpZiAocm9sbGJhY2tSZXNwLmRlcGxveW1lbnRJbmZvPy5zdGF0dXMgPT0gRGVwbG95bWVudFN0YXR1cy5TVUNDRUVERUQgfHxcbiAgICAgICAgICAgICAgICByb2xsYmFja1Jlc3AuZGVwbG95bWVudEluZm8/LnN0YXR1cyA9PSBEZXBsb3ltZW50U3RhdHVzLkZBSUxFRCB8fFxuICAgICAgICAgICAgICAgIHJvbGxiYWNrUmVzcC5kZXBsb3ltZW50SW5mbz8uc3RhdHVzID09IERlcGxveW1lbnRTdGF0dXMuU1RPUFBFRCkge1xuICAgICAgICAgICAgICAgIGxvZ2dlci5pbmZvKCdSb2xsYmFjayBpbiBmaW5hbCBzdGF0dXMnLCB7IGNvbXBsZXRlOiB0cnVlIH0pO1xuXG4gICAgICAgICAgICAgICAgcmV0dXJuIHsgSXNDb21wbGV0ZTogdHJ1ZSB9OyAvLyByb2xsYmFjayBmaW5pc2hlZCwgd2UncmUgZGVsZXRlZFxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgIGxvZ2dlci5pbmZvKCdXYWl0aW5nIGZvciBmaW5hbCBzdGF0dXMgZnJvbSBhIHJvbGxiYWNrJywgeyBjb21wbGV0ZTogZmFsc2UgfSk7XG5cbiAgICAgICAgICAgICAgcmV0dXJuIHsgSXNDb21wbGV0ZTogZmFsc2UgfTsgLy8gd2FpdGluZyBmb3Igcm9sbGJhY2tcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGxvZ2dlci5pbmZvKCdObyByb2xsYmFjayB0byB3YWl0IGZvcicsIHsgY29tcGxldGU6IHRydWUgfSk7XG5cbiAgICAgICAgICAgIHJldHVybiB7IElzQ29tcGxldGU6IHRydWUgfTtcbiAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgbG9nZ2VyLmluZm8oJ1dhaXRpbmcgZm9yIGZpbmFsIHN0YXR1cyBmcm9tIGRlcGxveW1lbnQnLCB7IGNvbXBsZXRlOiBmYWxzZSB9KTtcblxuICAgICAgICAgICAgcmV0dXJuIHsgSXNDb21wbGV0ZTogZmFsc2UgfTtcbiAgICAgICAgfVxuICAgICAgZGVmYXVsdDpcbiAgICAgICAgbG9nZ2VyLmVycm9yKCdVbmtub3duIHJlcXVlc3QgdHlwZScpO1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gcmVxdWVzdCB0eXBlOiAke2V2ZW50LlJlcXVlc3RUeXBlfWApO1xuICAgIH1cbiAgfSBjYXRjaCAoZSkge1xuICAgIGxvZ2dlci5lcnJvcignVW5hYmxlIHRvIGRldGVybWluZSBkZXBsb3ltZW50IHN0YXR1cycsIHsgZXJyb3I6IGUgYXMgRXJyb3IgfSk7XG4gICAgaWYgKGV2ZW50LlJlcXVlc3RUeXBlID09PSAnRGVsZXRlJykge1xuICAgICAgbG9nZ2VyLndhcm4oJ0lnbm9yaW5nIGVycm9yIC0gbm90aGluZyB0byBkbycsIHsgY29tcGxldGU6IHRydWUgfSk7XG5cbiAgICAgIHJldHVybiB7IElzQ29tcGxldGU6IHRydWUgfTtcbiAgICB9XG4gICAgdGhyb3cgZTtcbiAgfVxufSJdfQ==