import logging
import os
import re
from stat import ST_MODE
from unittest import TestCase

from stackifyapm.base import Client
from stackifyapm.conf import Config
from stackifyapm.conf import ConfigError
from stackifyapm.conf import IncrementalFileHandler
from stackifyapm.conf import RegexValidator
from stackifyapm.conf import setup_stackifyapm_logging
from stackifyapm.conf import StackifyFormatter
from stackifyapm.conf.constants import LOG_PATH


CONFIG = {
    "SERVICE_NAME": "service_name",
    "ENVIRONMENT": "production",
    "HOSTNAME": "sample_host",
    "FRAMEWORK_NAME": "framework",
    "FRAMEWORK_VERSION": "1.0",
    "APPLICATION_NAME": "sample_application",
    "BASE_DIR": "path/to/application/",
    "ASYNC_MODE": False,
    "CONFIG_FILE": 'path/to/stackify.json',
    "QUEUE": False,
}


class RegexValidatorTest(TestCase):

    def test_should_return_correct_value(self):
        regex = "^[a-zA-Z0-9 _-]+$"
        value = 'some_value'
        _validate = RegexValidator(regex)

        validated_value = _validate(value, 'SOME_KEY')

        assert validated_value == value

    def test_should_raise_exception(self):
        regex = "^[a-zA-Z0-9 _-]+$"
        value = '#$%^'
        _validate = RegexValidator(regex)

        with self.assertRaises(ConfigError) as context:
            _validate(value, 'SOME_KEY')

        assert 'does not match pattern' in context.exception.args[0]


class ConfigTest(TestCase):

    def test_config_creation(sself):
        config = Config(CONFIG)

        assert config.environment == CONFIG["ENVIRONMENT"]
        assert config.hostname == CONFIG["HOSTNAME"]
        assert config.framework_name == CONFIG["FRAMEWORK_NAME"]
        assert config.framework_version == CONFIG["FRAMEWORK_VERSION"]
        assert config.application_name == CONFIG["APPLICATION_NAME"]
        assert config.config_file == CONFIG["CONFIG_FILE"]
        assert config.base_dir == CONFIG["BASE_DIR"]
        assert config.async_mode is False
        assert config.queue is False

    def test_default_config(self):
        config = Config()

        assert config.application_name == 'Python Application'
        assert config.environment == 'Production'
        assert config.config_file == 'stackify.json'
        assert config.framework_name is None
        assert config.framework_version is None
        assert config.base_dir is None
        assert config.async_mode is True
        assert config.queue is True

    def test_environment_variables(self):
        os.environ["STACKIFY_TRANSPORT"] = 'agent_socket'
        os.environ["STACKIFY_TRANSPORT_HTTP_ENDPOINT"] = 'http://domain.test'

        config = Config(CONFIG)

        assert config.transport == 'agent_socket'
        assert config.http_endpoint == 'http://domain.test'

        del os.environ["STACKIFY_TRANSPORT"]
        del os.environ["STACKIFY_TRANSPORT_HTTP_ENDPOINT"]


class IncrementalFileHandlerTest(TestCase):

    def setUp(self):
        self.filename = 'somefile.log'
        self.filename1 = self.filename + '.1'

        self.remove_file(self.filename)
        self.remove_file(self.filename1)

    def tearDown(self):
        self.remove_file(self.filename)
        self.remove_file(self.filename1)

    def remove_file(self, filename):
        try:
            os.path.exists(filename) and os.remove(filename)
        except Exception:
            pass

    def test_file_should_be_created(self):
        IncrementalFileHandler(self.filename, maxBytes=100)

        assert os.path.exists(self.filename)
        assert os.path.getsize(self.filename) == 0

    def test_should_log(self):
        logger = logging.getLogger('test')
        logger.setLevel(logging.DEBUG)
        logger.addHandler(IncrementalFileHandler(self.filename, maxBytes=100))

        logger.debug('some data to log')

        assert not os.path.getsize(self.filename) == 0

    def test_should_rotate(self):
        logger = logging.getLogger('test')
        logger.setLevel(logging.DEBUG)
        logger.addHandler(IncrementalFileHandler(self.filename, maxBytes=20))

        logger.debug('some data to log')  # size 17
        logger.debug('some data to log')  # size will go over 20 bytes and will create new file .1

        assert os.path.exists(self.filename)
        assert os.path.exists(self.filename1)

    def test_should_recreate_log_file_if_deleted(self):
        logger = logging.getLogger('test')
        logger.setLevel(logging.DEBUG)
        logger.addHandler(IncrementalFileHandler(self.filename, maxBytes=100))

        # making sure the log file was created
        logger.debug('some data to log')
        assert os.path.exists(self.filename)

        # we cant delete file if still in use
        if not os.name == 'nt':
            # making sure the log file was delete
            self.remove_file(self.filename)
            assert not os.path.exists(self.filename)

        logger.debug('some data to log')
        assert os.path.exists(self.filename)


class SetupLoggingTest(TestCase):

    def test_log_file_should_be_created(self):
        client = Client(CONFIG)
        host_name = client.get_system_info().get("hostname")
        process_id = client.get_process_info().get("pid")
        filename = "{}{}#{}-1.log".format(LOG_PATH, host_name, process_id)

        setup_stackifyapm_logging(client)

        assert os.path.exists(filename)

        try:
            os.path.exists(filename) and os.remove(filename)
        except Exception:
            # we cant delete file in use on windows machine
            pass

    def test_should_log_777_permission(self):
        client = Client(CONFIG)
        host_name = client.get_system_info().get("hostname")
        process_id = client.get_process_info().get("pid")
        filename = "{}{}#{}-1.log".format(LOG_PATH, host_name, process_id)

        setup_stackifyapm_logging(client)

        assert oct(os.stat(filename)[ST_MODE])[-3:] == '777'

    def test_if_file_remove_log_777_permission_(self):
        client = Client(CONFIG)
        host_name = client.get_system_info().get("hostname")
        process_id = client.get_process_info().get("pid")
        filename = "{}{}#{}-1.log".format(LOG_PATH, host_name, process_id)

        logger = setup_stackifyapm_logging(client)

        assert os.path.exists(filename)

        try:
            os.remove(filename)
        except Exception:
            # we cant delete file in use on windows machine
            pass

        logger.debug("Test")
        assert oct(os.stat(filename)[ST_MODE])[-3:] == '777'

    def test_log_timestamp_format(self):

        client = Client(CONFIG)
        host_name = client.get_system_info().get("hostname")
        process_id = client.get_process_info().get("pid")
        filename = "{}{}#{}-1.log".format(LOG_PATH, host_name, process_id)

        logger = setup_stackifyapm_logging(client)

        assert os.path.exists(filename)

        logger.debug("Test")

        file = open(filename, "r")
        log_line = file.read()
        assert re.findall("\\d{4}-\\d{2}-\\d{2}, \\d{2}:\\d{2}:\\d{2}.\\d{6}> ", log_line)


class StackifyMsecFormatTest(TestCase):

    def test_should_msec_format(self):
        record = logging.makeLogRecord({'mgs': 'message'})
        datefmt = '%Y-%m-%d, %H:%M:%S.%f'
        stackify_formatter = StackifyFormatter()

        time_format = stackify_formatter.formatTime(record, datefmt)

        assert re.findall("\\d{4}-\\d{2}-\\d{2}, \\d{2}:\\d{2}:\\d{2}.\\d{6}", time_format)


class EnvironmentVariableTests(TestCase):
    def test_application_name(self):
        os.environ["STACKIFY_APPLICATION_NAME"] = 'test_name'

        config = Config(CONFIG)

        assert config.application_name == 'test_name'

        del os.environ["STACKIFY_APPLICATION_NAME"]

    def test_environment(self):
        os.environ["STACKIFY_ENVIRONMENT_NAME"] = 'test_environment'

        config = Config(CONFIG)

        assert config.environment == 'test_environment'

        del os.environ["STACKIFY_ENVIRONMENT_NAME"]

    def test_config_file(self):
        os.environ["STACKIFY_CONFIG_FILE"] = 'test_config_file'

        config = Config(CONFIG)

        assert config.config_file == 'test_config_file'

        del os.environ["STACKIFY_CONFIG_FILE"]

    def test_multiprocessing(self):
        os.environ["STACKIFY_MULTIPROCESSING"] = 'True'

        config = Config(CONFIG)

        assert config.multiprocessing is True

        del os.environ["STACKIFY_MULTIPROCESSING"]

    def test_rum_enabled(self):
        os.environ["STACKIFY_RUM"] = 'True'

        config = Config(CONFIG)

        assert config.rum_enabled is True

        del os.environ["STACKIFY_RUM"]

    def test_rum_auto_injection(self):
        os.environ["STACKIFY_RUM_AUTO_INJECT"] = 'True'

        config = Config(CONFIG)

        assert config.rum_auto_injection is True

        del os.environ["STACKIFY_RUM_AUTO_INJECT"]

    def test_socket_url(self):
        os.environ["STACKIFY_TRANSPORT_SOCKET_PATH"] = 'test_socket_path'

        config = Config(CONFIG)

        assert config.socket_url == 'test_socket_path'

        del os.environ["STACKIFY_TRANSPORT_SOCKET_PATH"]

    def test_transport(self):
        os.environ["STACKIFY_TRANSPORT"] = 'test_transport'

        config = Config(CONFIG)

        assert config.transport == 'test_transport'
        del os.environ["STACKIFY_TRANSPORT"]

    def test_log_path(self):
        os.environ["STACKIFY_TRANSPORT_LOG_PATH"] = 'test_transport_log_path'

        config = Config(CONFIG)

        assert config.log_path == 'test_transport_log_path'

        del os.environ["STACKIFY_TRANSPORT_LOG_PATH"]

    def test_debug(self):
        os.environ["STACKIFY_DEBUG"] = 'True'

        config = Config(CONFIG)

        assert config.debug is True

        del os.environ["STACKIFY_DEBUG"]

    def test_queue(self):
        os.environ["STACKIFY_QUEUE"] = 'False'

        config = Config(CONFIG)

        assert config.queue is False

        del os.environ["STACKIFY_QUEUE"]
