# Copyright 2018-2020 ejtrader, traderpedroso @ GitHub
# See LICENSE for details.

import pandas as pd
import pkg_resources

import random

from . import constant as cst


def resource_to_data(path_to_data):
    """
    This is an auxiliar function to read data from a given path, so as to wrap the load
    process of the static data files from ejtrader.

    Returns:
        :obj:`pandas.DataFrame` - data:
            This function returns a :obj:`pandas.DataFrame` object with all the static file's data
            retrieved from ejtrader.

    Raises:
        FileNotFoundError: raised if the static data file was not found.
        IOError: raised if the data file is empty or errored.

    """

    resource_package = 'ejtrader'
    resource_path = '/'.join(('/invest/resources', path_to_data))
    if pkg_resources.resource_exists(resource_package, resource_path):
        data = pd.read_csv(pkg_resources.resource_filename(resource_package, resource_path))
    else:
        raise FileNotFoundError("ERR#0115: data file not found or errored.")

    if data is None:
        raise IOError("ERR#0115: data file was empty or errored.")

    return data


def random_user_agent():
    """
    This function selects a random User-Agent from the User-Agent list, which is a constant
    variable that can be found at `ejtrader.utils.constant.USER_AGENTS`. User-Agents are used in
    order to avoid the limitations of the requests to Investing.com. The User-Agent is
    specified on the headers of the requests and is different for every request.

    Note that Investing.com, via changing the User-Agent on the headers of every request, allows
    a lot of requests, since it has been tested with over 10k consecutive requests without getting
    any HTTP error code from Investing.com.

    Returns:
        :obj:`str` - user_agent:
            The returned :obj:`str` is the name of a random User-Agent, which will be passed on the 
            headers of a request so to avoid restrictions due to the use of multiple requests from the 
            same User-Agent.
    
    """

    return str(random.choice(cst.USER_AGENTS))
