"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...options?.venvOptions,
            },
            pip: true,
            poetry: false,
            pytest: false,
            setuptools: true,
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "constructs",
            "aws-cdk-lib",
            "cdk-nag",
        ].forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientLanguages, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in __init__.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC_PATH = path.join(str(Path(__file__).absolute().parent), "${this.specDir}/${spec.parsedSpecFileName}")
SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Set up pytest manually since the default pytest generates tests for sample code which doesn't exist
        const pytestVersion = options.pytestOptions?.version || "6.2.1";
        this.addDevDependency(`pytest@${pytestVersion}`);
        this.testTask.exec("pytest");
        // Generate documentation if requested
        if ((options.documentationFormats ?? []).length > 0) {
            new docs_project_1.DocsProject({
                parent: this,
                outdir: path.join(this.generatedCodeDir, "documentation"),
                name: "docs",
                formats: [...new Set(options.documentationFormats)],
                specPath: spec.parsedSpecPath,
            });
        }
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.7.5" };
//# sourceMappingURL=data:application/json;base64,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