# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['sqlalchemy_declarative_extensions',
 'sqlalchemy_declarative_extensions.alembic',
 'sqlalchemy_declarative_extensions.dialects',
 'sqlalchemy_declarative_extensions.dialects.mysql',
 'sqlalchemy_declarative_extensions.dialects.postgresql',
 'sqlalchemy_declarative_extensions.dialects.sqlite',
 'sqlalchemy_declarative_extensions.grant',
 'sqlalchemy_declarative_extensions.role',
 'sqlalchemy_declarative_extensions.row',
 'sqlalchemy_declarative_extensions.schema',
 'sqlalchemy_declarative_extensions.view']

package_data = \
{'': ['*']}

install_requires = \
['sqlalchemy>=1.3']

extras_require = \
{':python_version < "3.11"': ['typing_extensions>=4.0'],
 'alembic': ['alembic>=1.0'],
 'parse': ['sqlglot']}

setup_kwargs = {
    'name': 'sqlalchemy-declarative-extensions',
    'version': '0.4.3',
    'description': 'Library to declare additional kinds of objects not natively supported by SqlAlchemy/Alembic.',
    'long_description': '# SqlAlchemy Declarative Extensions\n\n[![Actions Status](https://github.com/dancardin/sqlalchemy-declarative-extensions/workflows/test/badge.svg)](https://github.com/dancardin/sqlalchemy-declarative-extensions/actions) [![Coverage Status](https://coveralls.io/repos/github/DanCardin/sqlalchemy-declarative-extensions/badge.svg?branch=main)](https://coveralls.io/github/DanCardin/sqlalchemy-declarative-extensions?branch=main) [![Documentation Status](https://readthedocs.org/projects/sqlalchemy-declarative-extensions/badge/?version=latest)](https://sqlalchemy-declarative-extensions.readthedocs.io/en/latest/?badge=latest)\n\nSee the full documentation [here](https://sqlalchemy-declarative-extensions.readthedocs.io/en/latest/).\n\nAdds extensions to SqlAlchemy (and/or Alembic) which allows declaratively\nstating the existence of additional kinds of objects about your database\nnot natively supported by SqlAlchemy/Alembic.\n\nThis includes:\n\n- Schemas\n- Roles\n- Privileges\n- Rows (i.e. data)\n\nThe primary function(s) of this library include:\n\n- Registering onto the SqlAlchemy event system such that `metadata.create_all`\n  creates these objects.\n- (Optionally) Registers into Alembic such that `alembic revision --autogenerate`\n  automatically creates/updates/deletes declared objects.\n\n## Kitchen Sink Example Usage\n\n```python\nfrom sqlalchemy import Column, types, select\nfrom sqlalchemy.orm import as_declarative\nfrom sqlalchemy_declarative_extensions import (\n    declarative_database, Schemas, Roles, Grants, Rows, Row, Views, View, view\n)\nfrom sqlalchemy_declarative_extensions.dialects.postgresql import DefaultGrant, Role\n\n\n@declarative_database\n@as_declarative\nclass Base:\n    schemas = Schemas().are("example")\n    roles = Roles(ignore_unspecified=True).are(\n        Role("read", login=False),\n        Role(\n            "app",\n            in_roles=[\'read\']\n        ),\n    )\n    grants = Grants().are(\n        DefaultGrant.on_tables_in_schema("public", \'example\').grant("select", to="read"),\n        DefaultGrant.on_tables_in_schema("public").grant("insert", "update", "delete", to="write"),\n        DefaultGrant.on_sequences_in_schema("public").grant("usage", to="write"),\n    )\n    rows = Rows().are(\n        Row(\'foo\', id=1),\n    )\n    views = Views().are(View("low_foo", "select * from foo where i < 10"))\n\n\nclass Foo(Base):\n    __tablename__ = \'foo\'\n\n    id = Column(types.Integer(), primary_key=True)\n\n\n@view(Base)\nclass HighFoo:\n    __tablename__ = "high_foo"\n    __view__ = select(Foo.__table__).where(Foo.__table__.c.id >= 10)\n```\n\nNote, there is also support for declaring objects directly through the `MetaData` for\nusers not using sqlalchemy\'s declarative API.\n\n## Event Registration\n\nBy default, the above example does not automatically do anything. Depending on the context,\nyou can use one of two registration hooks: `register_sqlalchemy_events` or `register_alembic_events`.\n\n### `register_sqlalchemy_events`\n\nThis registers events in SqlAlchemy\'s event system such that a `metadata.create_all(...)` call\nwill create the declared database objects.\n\n```python\nfrom sqlalchemy_declarative_extensions import register_sqlalchemy_events\n\nmetadata = Base.metadata  # Given the example above.\nregister_sqlalchemy_events(metadata)\n# Which is equivalent to...\nregister_sqlalchemy_events(metadata, schemas=False, roles=False, grants=False, rows=False)\n```\n\nAll object types are opt in, and should be explicitly included in order to get registered.\n\nPractically, this is to avoid issues with testing. In **most** cases the `metadata.create_all` call\nwill be run in tests, a context where it\'s more likely that you dont **need** grants or grants,\nand where parallel test execution could lead to issues with role or schema creation, depending\non your setup.\n\n### `register_alembic_events`\n\nThis registers comparators in Alembic\'s registration system such that an `alembic revision --autogenerate`\ncommand will diff the existing database state against the declared database objects, and issue\nstatements to create/update/delete objects in order to match the declared state.\n\n```python\n# alembic\'s `env.py`\nfrom sqlalchemy_declarative_extensions import register_alembic_events\n\nregister_alembic_events()\n# Which is equivalent to...\nregister_sqlalchemy_events(schemas=True, roles=True, grants=True, rows=True)\n```\n\nAll object types are opt out but can be excluded.\n\nIn contrast to `register_sqlalchemy_events`, it\'s much more likely that you\'re declaring most of these\nobject types in order to have alembic track them\n\n## Database support\n\nIn principle, this library **can** absolutely support any database supported by SqlAlchemy,\nand capable of being introspected enough to support detection of different kinds of objects.\n\nIn reality, the implementations are going to be purely driven by actual usage. The\ncurrent maintainer(s) primarily use PostgreSQL and as such individual features for\nother databases will either suffer or lack implementation.\n\nAs much as possible, objects will be defined in a database-agnostic way, and the comparison\ninfrastructure should be the sole difference. However databases engines are not the same, and\ncertain kinds of objects, like GRANTs, are inherently database engine specific, and there\'s\nnot much common ground between a PostgreSQL grant and a MySQL one. As such, they will\ninclude database specific objects.\n\n## Alembic-utils\n\nCurrently, the set of supported declarative objects is largely non-overlapping with\n[Alembic-utils](https://github.com/olirice/alembic_utils). However in principle, there\'s\nno reason that objects supported by this library couldn\'t begin to overlap (functions,\ntriggers); and one might begin to question when to use which library.\n\nNote that where possible this library tries to support alembic-utils native objects\nas stand-ins for the objects defined in this library. For example, `alembic_utils.pg_view.PGView`\ncan be declared instead of a `sqlalchemy_declarative_extensions.View`, and we will internally\ncoerce it into the appropriate type. Hopefully this eases any transitional costs, or\nissues using one or the other library.\n\nAlembic utils:\n\n1. Is more directly tied to Alembic and specifically provides functionality for autogenerating\n   DDL for alembic, as the name might imply. It does **not** register into sqlalchemy\'s event\n   system.\n\n2. Requires one to explicitly find/include the objects one wants to track with alembic.\n\n3. Declares single, specific object instances (like a single, specific `PGGrantTable`). This\n   has the side-effect that it can only track included objects. It cannot, for example,\n   remove objects which should not exist due to their omission.\n\n4. In most cases, it appears to define a very "literal" interface (for example, `PGView` accepts\n   the whole view definition as a raw literal string), rather than attempting to either abstract\n   the objects or accept abstracted (like a `select` object) definition.\n\n5. Appears to only be interested in supporting PostgreSQL.\n\nBy contrast, this library:\n\n1. SqlAlchemy is the main dependency and registration point (Alembic is, in fact, an optional dependency).\n   The primary function of the library is to declare the underlying objects. And then registration into\n   sqlalchemy\'s event system, or registration into alembic\'s detection system are both optional features.\n\n2. Perhaps a technical detail, but this library registers the declaratively stated objects directly\n   on the metadata/declarative-base. This allows the library to automatically know the intended\n   state of the world, rather than needing to discover objects.\n\n3. The intended purpose of the supported objects is to declare what the state of the world **should**\n   look like. Therefore the function of this library includes the (optional) **removal** of objects\n   detected to exist which are not declared (much like alembic does for tables).\n\n4. As much as possible, this library provides more abstracted interfaces for defining objects.\n   This is particularly important for objects like roles/grants where not every operation is a create\n   or delete (in contrast to something like a view), where a raw SQL string makes it impossible to\n   diff two different a-like objects.\n\n5. Tries to define functionality in cross-dialect terms and only where required farm details out to\n   dialect-specific handlers. Not to claim that all dialects are treated equally (currently only\n   PostgreSQL has first-class support), but technically, there should be no reason we wouldn\'t support\n   any supportable dialect. Today SQLite (for whatever that\'s worth), and MySQL have **some** level\n   of support.\n',
    'author': 'Dan Cardin',
    'author_email': 'ddcardin@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/dancardin/sqlalchemy-declarative-extensions',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4',
}


setup(**setup_kwargs)
