import numpy as np 
import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns
import scipy
import scipy.misc
from scipy import ndimage
import matplotlib as mpl
import os

def norm(data, mode='max'):
    """Normalize 1d spectrum
                     - 'snv' - Divide by maximum.
                     - 'max' - Divide by maximum.
                     - 'area' - Divide by sum over absolute value.
                     - '01' - Scale to interval [0, 1] by subtracting offset before dividing by maximum.
    :returns: Normalized spectrum with same shape as y.
    """
    if mode == 'snv':
        m,n = data.shape
        return np.divide(np.transpose(data-np.mean(data.T)*np.ones(n)),(np.std(np.transpose(data.T)*np.ones(n))))
    if mode == 'max':
        return data / np.amax(data)
    if mode == '01':
        data = data - np.amin(data)
        return data / np.amax(data)
    if mode == 'area':
        return data / np.sum(np.absolute(data))

def coef(x,y):
    #calculation of pearson's correlation coefficient
    x_mean = np.mean(x,axis=0)
    y_mean = np.mean(y,axis=0)
    x_std = np.std(x,axis=0)
    y_std = np.std(y,axis=0)
    x_dif = x-x_mean
    y_dif = y-y_mean
    covar = np.sum(x_dif*y_dif)/394
    coef = covar/(x_std*y_std)
    return coef

def xy(spectra_data):
    """calculating the x and y dimensions of the spectral image using the spectra number - factor pairs"""
    x = spectra_data.shape[0]
    factor_pairs =  [i for i in range(1,x+1) if x%i==0]
    """#x is first column y is second column"""
    factor_pairs = np.array(factor_pairs)
    factor_pairs = factor_pairs.flatten()
    """the order of factor pairs dictate that the middle and -1 from middle are the correct x,y parameters"""
    xdimension = (factor_pairs.shape[0]/2)
    ydimension = ((factor_pairs.shape[0]/2)-1)
    ydimension = int(ydimension)
    xdimension = int(xdimension)
    y = factor_pairs[ydimension]
    x = factor_pairs[xdimension]
    xy = np.asarray([x,y])
    """x is first column y is second column"""
    return xy

def file_locator():
    filelocation = input('Where would you like to save your data and graphs? i.e. "/home/Desktop/DataFolder"')
    location = filelocation
    #asks what where would you like to save your RSI's
    return location

def image(data,xy):
    d = np.asarray(data, dtype=np.float32)
    d[d < 0.78] = 0
    im = np.reshape(d, (xy[0], xy[1]))
    rotated_img = ndimage.rotate(im,-90)
    flipped_img = np.flip(rotated_img,1)
    upper_labels = 100
    lower_labels = 10
    size = 500
    xticks = upper_labels if im.shape[0] >= size else lower_labels
    yticks = upper_labels if im.shape[0] >= size else lower_labels
    ax = sns.heatmap(flipped_img, vmax=1.0, vmin=0.0, cmap="RdBu_r", xticklabels=xticks, yticklabels=yticks, square="equal")
    plt.rcParams["font.family"] = "Times New Roman"
    plt.xlabel('μm')
    plt.ylabel('μm')
    return 

def polyrsi(data,xy,location):
    """This class runs a Pearson's correlation coefficient across the data set for the different plastics that are contained in the current database"""
    abs_ = np.array([0.324240310479811,0.502718684919147,0.501568080138429,0.502192225373969,0.505444821590308,0.515194267128613,0.522942332628866,0.530594206575332,0.534506319523412,0.534734367792486,0.533885814952004,0.537247974020064,0.545821847922961,0.564452325969305,0.589784463727637,0.617798700413119,0.641413906004946,0.660506919391227,0.669006859621460,0.661540857385508,0.645929971156884,0.623884916336890,0.608006840433507,0.603624954998117,0.617208451529075,0.645722301757183,0.691816920517478,0.739448442497717,0.788326529055274,0.836925701186132,0.893221923108955,0.964134038299296,1.05817507505124,1.19309392776878,1.42282796656662,1.87200518043447,2.85383798943657,4.29442434951253,5.72323345773765,6.34622492590528,6.33229053430971,5.39614875290524,3.94771133133173,2.60341027923243,1.64612015680670,1.16840683318900,0.947108477843587,0.868779021338873,0.888801991519813,0.991799000493389,1.17878538909270,1.45626519523877,1.78034163560012,2.05351873068061,2.20242450947526,2.18921329797566,2.02206087502713,1.75676666110591,1.46660048877121,1.21276809386672,1.02272036359941,0.889936569223197,0.807066505135908,0.748810236505905,0.710167864841976,0.685031653528867,0.675417816023304,0.678749976725559,0.689917027273262,0.701300663021401,0.719094721333454,0.738114676954264,0.757900842811968,0.774457909528400,0.790314111927721,0.805875281008769,0.830223457522739,0.865918219644279,0.911575049506191,0.947807721454829,0.956132304949143,0.938922707694111,0.899883094417941,0.851374790864813,0.809586476310158,0.780174333530644,0.757507619463528,0.741238666657412,0.723589469667329,0.698198227774440,0.667160546351637,0.635911165428580,0.606077551942776,0.579218258403686,0.557061113141747,0.534998962882038,0.515685946131342,0.497639408800449,0.482364907058726,0.468473000738709,0.455712135683608,0.443106217854695,0.435225200164791,0.427207280703146,0.422975330113979,0.423245897828074,0.426816023946091,0.437151100517480,0.452752422171154,0.472415793496239,0.495482859477172,0.525481861724520,0.570314366717900,0.637791194310127,0.735656179506569,0.862161248325067,0.992697260600159,1.08573668036807,1.11133141013873,1.07318143405517,1.00163924810671,0.924208032327324,0.863595949747629,0.825424816444475,0.808624046607418,0.810972556812284,0.840143603741798,0.913187116183977,1.03456698764678,1.19270708484339,1.36056451692666,1.49257930732748,1.56756901960279,1.59631646065656,1.59765064996807,1.58812126264694,1.59029360964477,1.60981774236348,1.64921651400665,1.70509585845419,1.76572985381939,1.81688630383464,1.85641757744053,1.89960747336745,1.95327815553690,2.00842163668988,2.03719912071429,2.00014570093008,1.87976927073951,1.68403335215832,1.45431107647642,1.23202165702633,1.03857927240064,0.890858391575998,0.787644073081898,0.710582296152724,0.651136572668731,0.600311160820939,0.563296695441837,0.533980395912251,0.510455691492682,0.493625614242518,0.482873168107980,0.475613466226126,0.472926770782382,0.471832992718161,0.473359681259586,0.479577757951924,0.492356273201882,0.512176830575982,0.536594763658191,0.565264339497350,0.606187534708599,0.659068558367932,0.720699235012645,0.786672345454860,0.862284476453533,0.959970866821538,1.10185901463755,1.29982701490744,1.54619335588926,1.80761317346961,2.02149594707258,2.16414812320343,2.22194007746905,2.20865144758848,2.15890352629460,2.09406996571195,2.01606265595333,1.91479303485369,1.79144123133040,1.65332712295077,1.52589606449165,1.42288154096530,1.34289989811533,1.27540992218974,1.21341117686049,1.15586374841048,1.11362365755436,1.08404999045416,1.07132612436545,1.06747517779487,1.06631329926608,1.06284140959092,1.05829305132635,1.05833850629757,1.06823588250951,1.10000120004047,1.16709324641866,1.27713398406219,1.41959542906900,1.56403401063012,1.67228732708349,1.70345760254701,1.65326507433486,1.54121829051772,1.39654146387539,1.24489305875710,1.10111907769174,0.973667416852634,0.871983090366829,0.798435630024041,0.757857399935081,0.741410008292728,0.740071169513447,0.739264551440866,0.734713781600160,0.720641885579084,0.703697685319059,0.679317706626728,0.646916900475322,0.611104999693682,0.569797152363903,0.531432206557272,0.507801287302377,0.498333457666759,0.500276401610242,0.511672642751371,0.528420552881705,0.548267723230350,0.559527812504979,0.559523044527141,0.546449145390925,0.523705782776736,0.495896096361248,0.470544898397827,0.449572443357346,0.435815562557444,0.425582071414886,0.419092767084770,0.416523190499829,0.412140225780667,0.413399160268648,0.420831571332481,0.435903075146511,0.457972686321504,0.486893728441476,0.518065657984027,0.544372502535975,0.561897809722902,0.570657287691239,0.579278657479936,0.591858819936047,0.613979015174410,0.647265874505484,0.687392068471335,0.737469541681103,0.794975077916267,0.859335853565331,0.921406633066290,0.973106496320952,1.00639143958053,1.01308528962952,0.992462240401397,0.952167831394273,0.892792274452052,0.828179394716480,0.756250211200179,0.689128058829486,0.630481940385704,0.584401972414541,0.548634015589272,0.523951579959247,0.506669012704482,0.498223878345678,0.500128566663651,0.512522616676851,0.526044479153223,0.538104098823433,0.546528442902721,0.549051177106778,0.542626832420632,0.531414396515671,0.519931107144784,0.516764726036827,0.520363355726566,0.530206460252557,0.539865925371573,0.555497549586844,0.578613333673390,0.611016660771620,0.647900409362747,0.682376281566151,0.705412786380351,0.719132006519810,0.723974053340024,0.728456698126126,0.733584707383975,0.747404424267730,0.772030620479556,0.814996317866839,0.872486091034194,0.952208662969092,1.06434216936201,1.22149326216854,1.44542593786828,1.76791376770229,2.25220682436472,2.89126713782657,3.57806491900478,4.08098767950238,4.23922987130840,4.11550657958182,3.65909024506741,3.00126273987059,2.37583676361328,1.91611279397883,1.61627431724032,1.42560655880637,1.30689779941844,1.21813262517605,1.14655221249355,1.08255557841349,1.02315151907058,0.969332103723357,0.923771747348372,0.882920619483853,0.851746465772519,0.830367917015068,0.818084076097059,0.813251280761807,0.822204457052053,0.853486794042897,0.903888066614511,0.970674115927896,1.02996119453068,1.05487187427822,1.03568218968840,0.987074116246838,0.929969196354645,0.901678523267635,0.921703899097005,1.00864582574743,1.15133465170626,1.33779606986673,1.52044140553676,1.64431110754338,1.67141087792940,1.59451005529712,1.43249304886052,1.22672521128536,1.02443026977726,0.857440852607865,0.729332305448129,0.644475973788010,0.591653915505493,0.555623281396610,0.527024651321031,0.504360267382018,0.483102271145741,0.466389478198948,0.453439059809076,0.445801226347511,0.443223819946601,0.448597675313577,0.456644561057048,0.470753870079600,0.487129505769780,0.508252967440867,0.530265758462915,0.551077447101486,0.571731446691972,0.589633923692824,0.604099515386243,0.615951231813405,0.622482329016558,0.626839327275430,0.629259201018635,0.634823653405010,0.636000917548717,0.654384998928648,0.700212709647725,0.785315110468874,0.639677524579313])
    data_ = [coef(i, y=abs_) for i in data]
    im = image(data_,xy)
    my_path = os.path.join(location)
    plt.savefig(my_path + 'ABS_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pa = np.array([0.751247572360682,0.885136639636111,0.926825803228622,0.940333693628585,0.968569060120002,1.01605542335959,1.08495058267878,1.17861308944274,1.30623177267302,1.46069538943026,1.65638390392114,1.86387472086868,2.06202283075647,2.19340460965490,2.23208657716249,2.18102492094778,2.02230107692739,1.76156632676747,1.45560432642149,1.16052765208942,0.911026073542800,0.704826871234973,0.536061710181924,0.409129477609568,0.320988559940166,0.259468624799870,0.218224506788652,0.190416222827705,0.175381965156611,0.178579462146454,0.194015227834162,0.220067616364235,0.256148033469779,0.291247996082332,0.328720255338034,0.362530263777558,0.388928179503354,0.407786857907796,0.419109197490107,0.422722556953754,0.423459019654349,0.427479609919505,0.433963066756998,0.445303826829957,0.464639276998287,0.490521664268319,0.527298286237531,0.572219872812232,0.620960363479866,0.673099893509908,0.727291473965244,0.786080628298556,0.846445541156260,0.913150294233034,0.988545490382438,1.08639315561384,1.20384835689689,1.35301612331450,1.52005523251140,1.68371358299623,1.83150558153721,1.94391468763632,2.02255314113605,2.07845988695600,2.12184485131957,2.17223447875988,2.23857162644653,2.30905374920012,2.35506875999636,2.34996348538195,2.28111460404580,2.12338417531789,1.90868703273913,1.67830311406148,1.47741339879255,1.31406992321478,1.18651832444947,1.09003275540261,1.02187440661699,0.979759692810801,0.967561038422428,0.984245276444248,1.02752441158168,1.09122990913957,1.16166905895079,1.22560011119027,1.26366599551290,1.27474598148924,1.26005070741487,1.23212207964476,1.20180919714868,1.17865978866819,1.16460189503787,1.16199775649602,1.17136864588362,1.19703785929407,1.24995561139970,1.34292411691089,1.50086328828989,1.75150828953956,2.08950978424590,2.44558132126809,2.64188487423576,2.65851680621886,2.55461068152493,2.23232658901232,1.84248525076384,1.53261256089072,1.33694999193279,1.22594320330098,1.15833628239600,1.11026924059973,1.05491640310168,0.976805467984138,0.867479666017092,0.717623637110752,0.548470996237589,0.389452974891639,0.264719122240502,0.187052727121916,0.141431609277593,0.123092482777505,0.124372581630390,0.140603449824954,0.170664144944057,0.211369198788309,0.265765516506119,0.323915413401916,0.372837075537511,0.403949094035533,0.410661171917432,0.392393470599834,0.358407716808135,0.320562527090342,0.291709956218907,0.275258331482695,0.269225172070142,0.269707684772772,0.274504647645583,0.281537488096874,0.290427440696686,0.303598604544997,0.319890046603155,0.340649329300368,0.367975544466963,0.406790153617057,0.452040562851230,0.497343918176016,0.540215552722512,0.581684748583554,0.631251345254443,0.695814921239430,0.791250995536724,0.929931815819471,1.10331598633474,1.28600637009521,1.42031836795802,1.48194357027615,1.47423346553009,1.39793633075614,1.26564642657588,1.11953447357137,0.995424959991952,0.901707283243537,0.837335004855856,0.795405632010986,0.767677494421134,0.748342951824070,0.734510604143247,0.728344641270415,0.725385972859180,0.724828086870760,0.730893723513840,0.740195631383710,0.750453636451364,0.763630788092953,0.779895001487423,0.803375270117855,0.835208253703340,0.875725213993835,0.926868741852192,0.995674156425668,1.09142297773154,1.22623313168511,1.41839820003886,1.67959061246358,2.01084956098038,2.37160677410931,2.69715810243613,2.91024079037794,2.99151477943825,2.97777301226866,2.86915663488080,2.68665403132908,2.47396124321522,2.25313624834509,2.04128235441699,1.82721859351318,1.61093682428855,1.40361288949616,1.20737308557640,1.03657026329232,0.900455898940382,0.793926383274922,0.716634563945875,0.653217910799050,0.605211994413946,0.569217951152867,0.549381775208862,0.548009165762822,0.562472251869042,0.583939791771093,0.601930995467995,0.607341318223476,0.591233236275227,0.555827139154110,0.511098277452359,0.464282980880654,0.426357676679679,0.398339038377676,0.387072397347669,0.386372832826077,0.397571886386031,0.416885420476944,0.442099912306301,0.472249614603557,0.501493926946586,0.533360354237113,0.570117981453478,0.614964953612763,0.662158383051536,0.708438735563438,0.749081210084081,0.782672369182640,0.807569953907708,0.823578557012017,0.824379512814201,0.804046740790607,0.753934336027285,0.675936283912084,0.573213760803502,0.458699030229820,0.349664263032414,0.255017259788710,0.189115174288210,0.144609215402267,0.121236359945699,0.116050015737073,0.127059043894514,0.151800738287017,0.194853802467966,0.257706808787453,0.352186269648645,0.482770685940081,0.651702128927131,0.847492005569218,1.07319192550379,1.31730928212865,1.57210336876754,1.83947401289827,2.12735946983715,2.45822287736454,2.83030243773844,3.25981604896298,3.70444592700345,4.12086480244975,4.42379675660751,4.53630083448596,4.52448573515952,4.36774893287087,4.03381189439111,3.66622317951933,3.33108750239567,3.04156274148957,2.77372365796683,2.50404100132594,2.23607939507555,2.00029798225482,1.79399463052428,1.62622907623981,1.49369766854293,1.39645653955133,1.34007541365516,1.32393304997823,1.34090573043699,1.37163496870041,1.38014825142438,1.34353032420010,1.25142853882540,1.09856927627137,0.922297345203065,0.761992899475088,0.633154497216929,0.541150926701299,0.476201702021976,0.431444372800572,0.401135762676869,0.378946706355708,0.364673343246474,0.354920022026669,0.347840538834966,0.339864530970304,0.329551212983821,0.318618564024897,0.309390582702651,0.302026027803124,0.294301352712648,0.289093179368598,0.284866316303586,0.279862550383828,0.278279137638094,0.279828121450221,0.282010901815164,0.283312770056087,0.285434960123538,0.289419117491215,0.293719719925966,0.299413405050841,0.306624624892285,0.316165989561084,0.327573696808803,0.338687774323460,0.346754327074491,0.353186930933908,0.358028475357598,0.362582747446936,0.366425910791759,0.371348860105602,0.372830716019168,0.371179714690819,0.369956295240132,0.366756533742717,0.363331211442361,0.358758128828075,0.352572548460454,0.348398538369583,0.342726595368008,0.335017479673889,0.326457343369171,0.317959739074092,0.310854601774093,0.306449180013150,0.302034632553420,0.297341254865044,0.291902115845775,0.287723953113895,0.283339014662406,0.282055793835666,0.283242692430434,0.284998442865133,0.289136330380953,0.296447077617628,0.304218925227475,0.311814015361162,0.322284556022749,0.334081102187487,0.350363709392791,0.369887504445920,0.392717690982042,0.419418717489081,0.449159293881861,0.483124749250347,0.526464402574838,0.579930643294307,0.645993858241379,0.728987701444816,0.837333067675747,0.977659425410818,1.15172734165941,1.39225648377906,1.71266862450977,2.11590090640123,2.53150437654033,2.84341174741809,2.95386165459703,2.94796025611102,2.80732955193642,2.53328337198757,2.23783301678303,1.95231399691629,1.70676392660181,1.49576319207171,1.32003388461085,1.16098527018785,1.02494740823461,0.908135862130173,0.801927816205848,0.706622158796115,0.618040860111017,0.533647181265041,0.455993370180952,0.388992914835770,0.317018847396506,0.181569652438237])
    data_ = [coef(i, y=pa) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PA_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pc = np.array([0.304761012558976,0.252335039297393,0.215591801597214,0.194531299458437,0.191106565270170,0.204272690332062,0.219602827817347,0.226764666863734,0.224074275276145,0.214871785552222,0.201301769752099,0.185231497695416,0.165288436734905,0.142357915994681,0.120983469208670,0.104566593641114,0.0937184166530162,0.0887803240260220,0.0858626219482413,0.0838641234460020,0.0821096427145179,0.0788970925676671,0.0753402861726614,0.0730442301974528,0.0695835529951969,0.0678508478613733,0.0663941882909196,0.0654385263999903,0.0649069797873346,0.0658659046191442,0.0673608156541614,0.0711074695667414,0.0773238110003052,0.0875572819467162,0.0996182883462137,0.112862048213771,0.124808255212413,0.137449098116851,0.153580243072885,0.176975315803529,0.204941466238471,0.230154015305753,0.240950568643563,0.232860201267555,0.208449285420986,0.179130280010297,0.155961519524829,0.146787748884730,0.149191132986120,0.154396864615479,0.154162609020720,0.143528901505195,0.123886418808230,0.102565399273314,0.0851481865455548,0.0727970541752933,0.0649022574672158,0.0593059540405599,0.0552665917808523,0.0519295531210816,0.0491333646465553,0.0476713517000986,0.0466443821541458,0.0453528209080910,0.0436573452690394,0.0413910021866214,0.0391064007143364,0.0380621155287821,0.0379763897942728,0.0382939016761315,0.0375645446577867,0.0371825597606632,0.0374460063587908,0.0383496892140243,0.0403439057527130,0.0442449075527394,0.0514488908790967,0.0639053183641652,0.0813590021735120,0.0980745426925201,0.108129886539695,0.108253788106266,0.100174838701280,0.0878870065828652,0.0802389296138075,0.0812779275024669,0.0933477431543192,0.119849755822822,0.162837840386791,0.222145404336337,0.305405896077995,0.426486898092876,0.587808507275891,0.769855554366593,0.925699742077141,1,0.961891162017528,0.826515989445222,0.640735867916335,0.455811172221655,0.305230141176757,0.199750359931499,0.131310264589132,0.0884558424082579,0.0624272059499733,0.0463375851020610,0.0369736739875464,0.0345843598730752,0.0388454391058450,0.0492709434841314,0.0655126591694761,0.0830360267537521,0.0970088056167994,0.103219067726979,0.100479810262571,0.0887289903303397,0.0742523735187948,0.0622654514351823,0.0578782285271786,0.0649849622678718,0.0834393485964020,0.108714841900727,0.135321402611095,0.158495551235659,0.181945497867031,0.219880873580064,0.290425803700421,0.404083737841007,0.538448017362311,0.647045248504652,0.682238138361707,0.625545137754221,0.501761882553540,0.365558407466384,0.259794627731403,0.201282009075544,0.177615734850545,0.171117834364104,0.166825494947656,0.160491474202987,0.152356964982738,0.143643498600092,0.134637444214535,0.129637773278006,0.132686566487370,0.148345634296897,0.176813067793561,0.216321761247566,0.264286186933083,0.316063642648371,0.368631792963254,0.423611190979442,0.479839384752344,0.538552438386720,0.598577212140778,0.654135069912507,0.699891442029562,0.732168646150553,0.748629998382460,0.750424402241710,0.740257595909374,0.720727298939683,0.692896919889497,0.655778972044580,0.609996939012231,0.556425573727500,0.496833931491457,0.434828395136480,0.371641212912819,0.310628619082180,0.254035791327633,0.205164606433243,0.163536015027388,0.131082016966791,0.105771807847487,0.0874777345399387,0.0742575376517449,0.0660672112614981,0.0608577159878839,0.0593584565394184,0.0614656218412748,0.0665681029460780,0.0746009656288408,0.0862657425932377,0.0998251318205750,0.113888346599494,0.127210238138600,0.137687070444564,0.143976657730055,0.145201699271869,0.140740932446867,0.132082492857530,0.121074863105607,0.111580373379605,0.108162517211326,0.113459853521474,0.125738039867197,0.139558284715485,0.145472291815131,0.137687865204223,0.117807829504469,0.0929576385459879,0.0692921740776158,0.0529770007754428,0.0435871590240854,0.0393960545191126,0.0376875749776815,0.0371462408288513,0.0363080957858262,0.0365110665187636,0.0370699532793347,0.0384776186796430,0.0395361763140302,0.0406078430360789,0.0416889904870440,0.0438137741078697,0.0454726279221222,0.0468332432119218,0.0473289262703570,0.0485367953993248,0.0489625961818200,0.0500844473923569,0.0507177764456752,0.0524557231663037,0.0559084052235811,0.0609802569787180,0.0653434556314524,0.0678954657858426,0.0675775820052418,0.0667210154766583,0.0646887269141593,0.0628139837115638,0.0618803679292260,0.0618910098608084,0.0631741607830093,0.0675194324073904,0.0736140476921917,0.0815917922593578,0.0882281088961371,0.0916486195659825,0.0900216967847046,0.0845098322906168,0.0781305514130743,0.0730113450049718,0.0702674572664229,0.0701926530706892,0.0718237136067848,0.0752217275852959,0.0799571704923374,0.0854364970867621,0.0907302995154598,0.0942891963753704,0.0948531840430599,0.0927610333775343,0.0894191644591141,0.0859918880752410,0.0827818853247369,0.0804871955336584,0.0789415842444819,0.0777857246516559,0.0778687368149714,0.0789459840552808,0.0811354873027708,0.0842493491795789,0.0899346343358515,0.0993475171090638,0.114473708902677,0.136864333830721,0.163789683041807,0.190112064578933,0.209102950774257,0.216302808004257,0.211965373223696,0.200311486905008,0.186265678607585,0.176741577468810,0.173896704494775,0.181121565850526,0.197227054727864,0.218295484589436,0.235422987094838,0.242927407129004,0.236608285683673,0.218880720639935,0.194565857822613,0.169790942176483,0.146707203122389,0.127565093134889,0.112372052203686,0.100492710526657,0.0915878329566145,0.0852822774270681,0.0802752889451319,0.0769192077415774,0.0739634948243892,0.0728807580711850,0.0725969343127567,0.0733795018973540,0.0743824897026959,0.0768917078609320,0.0802490618245017,0.0842752481402379,0.0889610173123972,0.0937072380742361,0.0966859346225824,0.0974138333703794,0.0949602851709518,0.0895591357144677,0.0834829357278089,0.0775075866911879,0.0724022561992347,0.0692957689408826,0.0666163626498176,0.0653385988869296,0.0636846591903848,0.0622583926401932,0.0608490817530512,0.0606238004194495,0.0606423007512234,0.0608929990533682,0.0600030340590358,0.0596436705099394,0.0589647520081080,0.0586644889041990,0.0586370294698561,0.0584633749418443,0.0577462668498985,0.0576324019703552,0.0574213456259596,0.0574382182323792,0.0568126520956269,0.0570378171062387,0.0567151926826796,0.0573053360548562,0.0582794405089907,0.0605265218777964,0.0624436767221481,0.0659614219097049,0.0689607847065459,0.0727916904025107,0.0777185739458830,0.0842031067171873,0.0920191102423627,0.103929070619512,0.119641411856761,0.142018527555006,0.173802973590721,0.216538709954431,0.270956206483090,0.335024668714103,0.399124954732471,0.460151175415579,0.515434994365359,0.570308579712709,0.631581643771978,0.699604978607004,0.762421605626169,0.804105057015078,0.805256861325244,0.760166331380248,0.675436636418690,0.569436620978920,0.462271971362029,0.368164155921577,0.293701285127252,0.239080913990560,0.199576251468070,0.171139886997605,0.149488619527853,0.132312904837582,0.118910618792373,0.108153192163762,0.100237090196857,0.0939878722697680,0.0881213911123398,0.0837790978255320,0.0791805818979259,0.0747005035880865,0.0707695618431619,0.0664328954504084,0.0629537971955845,0.0602397335247599,0.0569037439396587,0.0549994725841523,0.0533576740867563,0.0513899131651253,0.0501065941502636,0.0490664413296692,0.0473963904512269,0.0465564749289163,0.0458804965765482,0.0457172723021972,0.0459352376847962,0.0465343927243454])
    data_ = [coef(i, y=pc) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PC_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    cel = np.array([0.825419537454669,0.841426877066403,0.787008869580240,0.765966928861728,0.788499236543265,0.907868408756628,0.880110136464936,1.00002662042256,0.829400800274424,0.893500833123164,0.777034433334547,0.841684917052899,0.974089380470051,0.884512469829704,1.04530212664326,0.917662237686090,0.886201215225494,0.998457630658538,0.888489001327872,0.895364833793085,0.980089518602852,1.01759330948766,0.918022058712228,0.912747449881122,0.948611195998743,0.885845064787386,0.790173594472916,0.940668649910686,0.962454284259787,0.902512627325427,0.917288699205629,0.967877727960442,0.976126847603774,0.935655009060607,0.951598659277822,0.899606505449444,1.05508061325675,1.00650714032746,0.975264677444748,1.02229964226307,0.829424129894167,1.00012527094934,0.828381512081093,0.977612997003243,0.941241144142932,0.994575608266590,0.950728704465885,1.03755280170698,0.935661956155611,0.990505550947529,0.997687400673076,0.909234852069791,1.07097369893275,0.891431043498781,1.06828839743568,1.07561944747489,1.00013021657625,0.960230233767040,1.00743659996155,0.938239064575869,0.870825545678855,0.952541753317877,0.932846107803306,0.945792893907633,0.918311001702712,0.931609531042114,0.941101524461117,0.850238366221953,0.977181652942905,0.941555625568549,0.933780744008790,0.919425475111608,0.902626677398703,1.00409135977805,0.938818834021120,1.07029204836833,0.982276455287475,0.968891097064256,1.03214504678308,1.02937943424521,1.09237080894910,1.03096467625573,1.00467380886489,1.15949902900509,1.08519063240734,1.17092097317863,1.12118968807620,1.24342131593099,1.14912332343598,1.22960092442011,1.10182846975927,1.09098196100572,1.15608205188980,1.19089718359420,1.03934755431772,1.00812269614259,1.03652162602898,1.04009445851706,0.978354432400008,1.06916542335124,1.06996911350133,0.958071423146025,0.967769678253932,1.04786579083197,0.961374052883476,0.969159342041216,0.902182884267352,0.928436671461107,0.855723997557011,0.943839474951665,0.929907832306170,0.849132968746376,0.916199495310885,0.912552316634148,0.956416597539105,0.948173995557853,0.995500872739560,0.947793349287655,0.836404052351445,0.891697887573752,0.952471138730913,1.02643724119851,1.10795733613160,0.957184276735724,0.965320060614412,1.06655438253668,0.833846886953005,0.869673213301443,0.897759246180474,0.925585218380173,0.893855754118658,0.895994696955632,0.924184671377068,0.890798491793162,0.986900813096195,0.967300523180362,0.990450352013385,0.959325661991424,0.988928994309709,1.06154507528258,1.00464480119236,0.969292552047340,1.03107660039631,1.03083434954657,0.922990061629507,0.733732871985641,0.619992923649522,0.608905814719275,0.396658133859155,0.540096799795439,0.473128702080072,0.475976788793027,0.564762398841685,0.436401254937082,0.495513342256450,0.476901628328145,0.419108946849369,0.787835124361956,0.808191899307006,0.929200560929375,0.967306851676595,1.08427053282814,1.15816887107783,1.05576046185505,1.13019301228503,1.06235286640725,1.09880515769761,1.04939018767245,1.03565706191759,1.09157223046622,1.10478241820324,1.19377636084192,1.20826696278624,1.01241499233880,1.15239025953152,1.34339851788126,1.44637299786594,1.35622120047210,1.52940946929331,1.57303387558703,1.61130889594825,1.76251163791337,1.67557361280857,1.66567064495849,1.80366058696439,1.77878002912466,1.82502238823035,1.73249288973229,1.71425958125498,1.72003353536890,1.62603409232416,1.67027265177651,1.60402976940495,1.50872943306480,1.47406743831224,1.48401366697223,1.51026832769745,1.40252187978260,1.49336820972747,1.53799954036460,1.49587791890950,1.41513382590287,1.44214189911170,1.39252493625262,1.46568112916695,1.40997519455150,1.40328159930315,1.34973082637587,1.33830222260487,1.19250711037978,1.27377537016090,1.31337132838752,1.25430084435482,1.27813488975246,1.24760606667565,1.23244236126052,1.27835079254532,1.22086312891505,1.18224674397654,1.15649731824132,1.12815411689348,1.14318937517964,1.20315039765591,1.04036008781506,1.10237991595516,1.11286042826908,1.15535558265968,1.08989531884389,1.05221902514179,1.01323203187335,0.970068896253666,1.04696408141253,0.944722264832819,0.836181596472371,0.844749067199151,0.898293501566806,0.803333566710527,0.819650613755770,0.918625368018728,0.696253098930930,0.740337994166050,0.740966208047318,0.730496179663016,0.830357739427260,0.735319333968879,0.689727056302964,0.655603888503448,0.728781137186619,0.691775193740249,0.851073725781997,0.752115298648497,0.714263332475223,0.656812036135912,0.645792225276703,0.610043493170970,0.737100359194845,0.768443046066170,0.778684310046146,0.682675167697339,0.617819849159641,0.722512539075532,0.662315994947391,0.748853310198787,0.686380663405647,0.741583235031838,0.764945311299511,0.844186503467182,0.682001578477424,0.753999973531941,0.749717369548004,0.745726777469209,0.777876194965069,0.705410069193701,0.725426306502066,0.737794649422058,0.685809073265184,0.772068934376027,0.784152756437139,0.661959764386025,0.714553459233801,0.733558536229175,0.751211631904750,0.750342529994491,0.667548110287938,0.659536699393947,0.764987744736586,0.804560393947706,0.605619554647135,0.762938975325957,0.765717374711690,0.756526544397442,0.663446674476808,0.643203469427939,0.713135256243991,0.702631330352909,0.764081274234777,0.903718671441350,0.825046949858238,0.757977822369098,0.755957911116977,0.800931618818393,0.736014537059645,0.879513298657625,0.704756546445062,0.828376617248893,0.782609217097007,0.806133901002672,0.768500573117080,0.853877748712364,0.750016429088155,0.839695199621468,0.831076222513535,0.810672506937287,0.798004439603303,0.717202082650190,0.809575067946220,0.912978817432288,0.907985436358540,0.758665349691710,0.873061244128593,0.836378733188504,0.904121409157979,0.797987578648446,0.876816381806690,0.889306193407987,0.968180032671473,0.938715057984016,0.940457773308558,0.992330084650518,0.938498135921402,0.874196448968785,1.04777593628934,0.988802836817355,0.980167292612561,1.08766916809141,1.01356968136557,1.07003287935606,1.16148412317089,1.04001988680166,1.22162554824125,1.14686128506955,0.981663069514751,1.18557361781189,1.32892807348280,1.29430577546861,1.34875374096851,1.27674616245727,1.35639274291764,1.43911411392090,1.45625724947924,1.51222647062782,1.45263566396545,1.36437840507629,1.45785737095267,1.51683194336117,1.57213877783415,1.56731194843065,1.65246220394300,1.75792330207257,1.81680949479361,1.90055870142815,2.18484978230710,2.37138003109761,2.25070275606479,2.05174737576647,1.86624386957365,1.59391836130126,1.50412779322534,1.36515060662074,1.04673915335054,0.915948905540246,0.875631693411579,0.813414044561359,0.706826931490106,0.714405175396402,0.647163400427145,0.816841798402843,0.781509350296610,0.776081524706568,0.789855592342366,0.774788472348664,0.756718105428533,0.791725568184123,0.716588271124246,0.663600450211525,0.738731192325203,0.681208283754488,0.637967641909690,0.585681847978396,0.666715940839003,0.592573648164798,0.614913661354351,0.554407622357039,0.782388148133738])
    data_ = [coef(i, y=cel) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'Cellulose_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    cib = np.array([0.225124389799921,0.226425206346971,0.233014497664444,0.244892263752340,0.262096014535908,0.286721122002594,0.295109440260484,0.299984937029028,0.303498820756918,0.298573034783379,0.306056935530200,0.315691711792963,0.309847443854161,0.304743317066275,0.299189304990635,0.295713208761205,0.302546170124933,0.305430138920052,0.308422198295072,0.312245932520323,0.336633866069595,0.379650557648305,0.427817306652659,0.457279168624819,0.473667186778841,0.460600220764640,0.437284595807094,0.394934150413668,0.344554234248615,0.308053944593638,0.286051435155986,0.279018222145291,0.284071726707229,0.278048089161603,0.276899405535340,0.268102465277902,0.266930171716303,0.272038951455065,0.281459187439437,0.290885890220233,0.307696201459969,0.306997133894304,0.302697159091336,0.290713448892693,0.290570799700899,0.286847163591013,0.298160418113923,0.306032161683705,0.299639602227859,0.293094242197739,0.292957524618631,0.289608371931323,0.292553558371067,0.298148555043462,0.296466770170843,0.291650702974548,0.285424856017610,0.292146138117805,0.292486866994412,0.293073179909588,0.300394777175018,0.303669238666840,0.300327592579311,0.303777434161772,0.294000676134380,0.281665056929112,0.284423011826591,0.291018608605113,0.297582564576677,0.298506322712435,0.287376621850537,0.283164405220954,0.282242271681882,0.281228026666084,0.286339612740687,0.288410761796987,0.280405062550860,0.278827827306099,0.278994029113451,0.270751199246768,0.281403642051601,0.297164610520730,0.326297672882171,0.340288326841754,0.364071181266011,0.359703876168308,0.349352375481336,0.344769892701229,0.349682005105868,0.338739781827096,0.332932936280496,0.314199806262348,0.302481299940914,0.310359238332088,0.322908541847101,0.346621943541565,0.364914872668538,0.390526176399752,0.416575923729741,0.422406926218137,0.412898276088485,0.374640429453123,0.343282913470281,0.305647055097333,0.290830337442534,0.293115179077011,0.290403792220429,0.278160923673992,0.279650047828276,0.270920492141675,0.270447003597863,0.279530679313069,0.270848997503100,0.266337756137333,0.264025857058031,0.267352641968159,0.280433784807511,0.321310591665869,0.368766228625863,0.410397437301458,0.439674980112234,0.443076186002701,0.428813291263796,0.381406958484121,0.353531951199127,0.320956531256547,0.298582876789787,0.303239072037955,0.305808882535133,0.287023896714715,0.291224718940175,0.285619421214516,0.259421355918646,0.252814368677548,0.246224640950584,0.236689026407377,0.245143457958901,0.256504533676228,0.255106637778349,0.265467416407305,0.278930332398033,0.288731403493561,0.316327139547899,0.333777323963108,0.333060253591074,0.326621141291656,0.297716729059404,0.271504703037120,0.266668870824116,0.271424985448283,0.275666471783000,0.289606599636914,0.278013050186538,0.257708665297735,0.241909038641097,0.249200042784501,0.271136174603165,0.329282286607056,0.433053472985307,0.561112711088843,0.660841204827033,0.712218173594864,0.668370152365379,0.569378122634705,0.472235948835252,0.396397788754374,0.342391923646334,0.313200346987937,0.298406059002488,0.287635077271689,0.282398568152437,0.295551089287695,0.303610537550400,0.313762237527208,0.313386750761847,0.302664162044109,0.309349696504904,0.347998551001521,0.390361595422924,0.466668483536270,0.536773595687988,0.581896075685325,0.610299804955620,0.599423406915631,0.570667224918591,0.566813955200292,0.591448826139106,0.598896679612430,0.598151564804110,0.568646628653618,0.519474322146453,0.492207188523901,0.483688452808124,0.481883890266652,0.472128195936512,0.457501600341767,0.437300465789085,0.445672256197992,0.445824873846049,0.453340611216102,0.446438175551523,0.440639054252091,0.427817230139787,0.443891180774522,0.449747303578366,0.464697701672144,0.476201601025407,0.496006668803079,0.504490315433050,0.517415492174652,0.504730937519072,0.498044827885774,0.484556661075075,0.468963684930427,0.458094545699735,0.441976462105583,0.424168484931358,0.398390982214884,0.393603239751712,0.391996198111486,0.396495451477372,0.415964485214258,0.451885005874729,0.478153425955840,0.530905258251797,0.631052576008973,0.766728096070741,0.908860138637133,1,0.974503932759979,0.829495483774547,0.657988305483037,0.499001258725535,0.408441713854502,0.363261396464082,0.328534518776082,0.310431130040825,0.297035223278891,0.286487387672038,0.278733649819349,0.275671570961028,0.264898489255124,0.262430842674228,0.255001078226260,0.246199141140936,0.239927802120991,0.244114854345635,0.248686024975289,0.254889555767272,0.281303306973600,0.302144576473687,0.337440835156654,0.370096028152207,0.409624432235096,0.456563126123461,0.484738761068727,0.471591402317024,0.429446618143712,0.380705160799227,0.328063759777899,0.304737391019344,0.285644886673374,0.273958422994472,0.268530537862382,0.256281799049949,0.222146060132285,0.210065878166361,0.213684614653455,0.226117633208549,0.234457160990537,0.219813715889602,0.194407440738239,0.184513101259904,0.194696727603924,0.218438833862623,0.258138578858492,0.286175415823190,0.312563399832707,0.310329576774107,0.281069139509246,0.252294337105589,0.226764004983421,0.212636869065703,0.217415381498311,0.210551800692481,0.206473339081401,0.212578176601666,0.208945487233355,0.223089186849496,0.227248222425991,0.230192907731057,0.250515129989932,0.269766987356734,0.287639527288545,0.318711566838527,0.323336265687305,0.328017678704330,0.321754152773995,0.291725706583928,0.273439639233274,0.277704886530870,0.275417624874642,0.298441914305979,0.323035331350780,0.350438219052853,0.390126727201788,0.434281993284285,0.431358863885622,0.421761634773155,0.396520504001204,0.364987126682945,0.331335166850870,0.315043942788564,0.277676089143130,0.267023410862712,0.274503048204431,0.274917564239520,0.282798513889937,0.294304040718753,0.285915482517636,0.281711779111430,0.296936336158200,0.311737646024563,0.328146503343830,0.344399145807913,0.337943499403557,0.334489873190207,0.347985119213383,0.365934979497535,0.395294976895211,0.442504641798586,0.493024059210322,0.553711338135139,0.602238905834983,0.620860390458423,0.616402066409634,0.577885823944677,0.556499573892342,0.547644059489451,0.556596106510981,0.575496748242868,0.602567019466492,0.602003783246487,0.597033746702843,0.573986530390230,0.546815798572384,0.524310815373889,0.521429816194957,0.513060195827273,0.505843764058301,0.493667480938769,0.486846248704736,0.487158028753108,0.504529095628356,0.517657938078977,0.539710154952413,0.559031987182837,0.568363352688589,0.573520589372191,0.571815470052573,0.563854150579961,0.561461997338335,0.573743124506100,0.575731037411414,0.582179356517860,0.570853554143478,0.544737135231186,0.499654844826456,0.448228769315094,0.399542282956944,0.362801941726960,0.338008549031910,0.326593748168524,0.312050876497854,0.290592148372406,0.280632104193053,0.271801061239143,0.261804514642530,0.251098521812070,0.244122347933456,0.237662253050729,0.246095632000085,0.241083428325685,0.236552347280588,0.226557177136763,0.215773871789528,0.211352454194093,0.231806942682102,0.234137710876249,0.233216658048748,0.228208670548894,0.238449775586016,0.253679117227744,0.273896695474078])
    data_ = [coef(i, y=cib) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'Cibacron Brilliant Red 3B-A_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pe = np.array([0.565711965585248,0.801933162378336,0.799908931233218,0.801444186172361,0.809398034681643,0.817049154961768,0.822496954046181,0.826244611111606,0.821423726578162,0.814332250197174,0.806195578214090,0.798318532958031,0.799202948917843,0.794080419559190,0.791795613993850,0.785416701527709,0.783622108519176,0.776706681659553,0.773890973674523,0.772590427048428,0.770193903450105,0.762542915800014,0.756045235107325,0.739591116812522,0.732253896303945,0.723953611310907,0.721305953926345,0.722137858362867,0.726205900515381,0.728142027842107,0.730953581751853,0.734826421515962,0.734564010135152,0.734126378409624,0.732695313290320,0.733417025944823,0.741637713683252,0.747026190750180,0.754255150057053,0.761891987869550,0.762304640242685,0.760311414815408,0.766148354294429,0.776147986845499,0.786900624017420,0.798045254264263,0.802221671736469,0.804638610361684,0.800724676142377,0.803423232270634,0.811552625541852,0.824476453055083,0.832077663718190,0.835937186000550,0.837747801478145,0.835532049508938,0.842799838729972,0.856258073762102,0.875138596860299,0.892894320189830,0.916372877582863,0.949049279976720,1.00071293360386,1.09853710734732,1.28889979139301,1.61843756316892,2.13257807466722,2.79849392416780,3.35568094565045,3.55255341098769,3.42975428638328,2.95145148563156,2.30141869139260,1.79781703162457,1.50321771371073,1.36495897546989,1.32512414449883,1.33534433128861,1.36584309998161,1.40352445843040,1.42509685988428,1.41267057298986,1.35761918960353,1.26367427923912,1.15807919623743,1.06281207231320,0.989286072556545,0.936704300080462,0.901385607374195,0.876993852258257,0.862187228100596,0.857264298044368,0.857687006127771,0.865840193002113,0.889563478175165,0.932374060996061,1.00236688676567,1.11739007382349,1.31784093190667,1.64413400477455,2.12851572856962,2.67971020166936,3.05385070878912,3.12269584923272,2.90887434178195,2.39632395953006,1.83379568453652,1.41145589780816,1.14664612411523,0.990443562136627,0.897900092586655,0.842192934810162,0.799961679975365,0.770594316958902,0.753724091751087,0.743650973946485,0.735027110060140,0.726526594121800,0.726415821408527,0.738872775401589,0.774787351084032,0.840421969202479,0.923778348712072,0.992610690096841,1.01619876619790,0.976722092322436,0.893743729640232,0.800352361396914,0.725187430563800,0.675377553408438,0.650062002630703,0.639056657646224,0.631627621550328,0.628742034872263,0.628243716450139,0.627871049355028,0.627725470006215,0.629219163679099,0.635503391671355,0.642391280726248,0.653099382524849,0.659803782345890,0.655709519940484,0.648869367815705,0.641691400100516,0.638565545727735,0.642960274671031,0.648379586701429,0.654079381938150,0.652344289838698,0.649344329858512,0.642975849407659,0.639834006692518,0.639398290503718,0.640805099722883,0.640549751980161,0.645475741386791,0.655896384343548,0.657189070335620,0.657924303946560,0.656779149089750,0.655600367335752,0.657560227531590,0.661791203135223,0.668458458364708,0.673804133611156,0.682708319023725,0.689871253830615,0.702105574601850,0.715027706742968,0.733334191377226,0.751080447268557,0.773898084660194,0.800583492951982,0.831522833555941,0.861839354738256,0.883826299892392,0.904055803921842,0.918211967317896,0.935320107706474,0.956512149340075,0.987910154730198,1.02338669277629,1.08549475226880,1.22869163395867,1.53033555815787,2.19379375420887,3.22413070096438,4.27539193102152,4.67706321178458,4.63760868566980,3.94275701418440,2.98912049200699,2.25578032762223,1.87536403884473,1.66342557823052,1.54247771182653,1.44084372556717,1.34764731289830,1.26022521230267,1.18044620819047,1.10302859836770,1.03006914481805,0.968267883802039,0.913370362690694,0.864967071874820,0.818592005187881,0.776432114472451,0.737709439442515,0.701481730952442,0.675764124109495,0.661392227409157,0.652456490438388,0.643925423241264,0.642346594806356,0.642670671547442,0.642635023721705,0.645256981880511,0.645203913275450,0.641063690161150,0.635315816286242,0.632649373224464,0.633812955080341,0.642668083519368,0.662177431624157,0.690345238960230,0.724900061975792,0.759121148338044,0.786220943458055,0.789443645468736,0.767858817241398,0.717398167877651,0.656878808818415,0.601209450207095,0.560045611857888,0.535676717253245,0.518195412014877,0.501531542358924,0.493518409340651,0.483038863555117,0.476273860406348,0.479001761192182,0.482454144307674,0.490898006562797,0.496988404451598,0.503864107159549,0.519517177669650,0.537195423624767,0.564984466863026,0.619319527432737,0.736081616516763,0.936404945210089,1.20265090001203,1.47972408124186,1.66610611010525,1.71543349160698,1.64093916501753,1.52059511483470,1.44217278082566,1.46358798365663,1.59623152593806,1.83916704589031,2.18644710367225,2.63761880625570,3.14602915491582,3.58370281806129,3.84574973253598,3.87149936855773,3.67996278800976,3.33331744266029,2.98184764766012,2.72203317313951,2.56791852569769,2.49642376476442,2.48110012748625,2.48904831154583,2.50465532421991,2.50882350699953,2.48441134624652,2.41722491876948,2.29891378556848,2.14087628440192,1.94958229754538,1.73876880648707,1.52549946211080,1.33368468019126,1.17631764130634,1.05710450330570,0.961142412590228,0.885080721016549,0.834093153239366,0.797170611416176,0.763348584803506,0.739909606295106,0.723485831051927,0.708267927481008,0.694814077484622,0.681726222237171,0.671187105903713,0.662266958184640,0.658462918089445,0.653845888560803,0.650417183134281,0.646149020481572,0.645735362990863,0.652074711772718,0.658443627205927,0.663475808282728,0.665334542750941,0.664012730406927,0.662271505590701,0.667229536608942,0.669184676101968,0.667581465390864,0.668088291751334,0.669207593772548,0.673849942104668,0.676079133003891,0.679189431435404,0.681480205609929,0.684043505689812,0.687158404991881,0.695253237775733,0.701175148001519,0.703188541964488,0.707436297085984,0.712030046834288,0.713966109036236,0.715877324883283,0.719745210714728,0.720903208813249,0.721538095028597,0.726043360506147,0.726793300003671,0.728411861621691,0.730331978498722,0.732927991381706,0.732628998035554,0.731975409139925,0.732931225244705,0.731403835154900,0.736464210442949,0.745886139671568,0.750452932385365,0.753511860873734,0.754907965547024,0.758006423109379,0.761609148700626,0.763505033722865,0.759843505701854,0.758588784889592,0.759272946651312,0.762458720369828,0.762192806333130,0.764282214515367,0.764634078558275,0.761807564982556,0.758727714145001,0.761630267201353,0.760719047294481,0.760344456145673,0.756139524665518,0.752326598270343,0.747499593841110,0.747176534174643,0.741652979396932,0.738196992450520,0.731373177979963,0.729298263108166,0.728454990577447,0.721629666857008,0.716167664803527,0.711484456697190,0.710608129815808,0.711599588543333,0.716713630704699,0.719938171449089,0.720623944547269,0.722632416466660,0.725164343413109,0.736527631038410,0.743388990028357,0.754791441380892,0.760034275074368,0.759731201294259,0.756253752062741,0.755890761644267,0.756823689616522,0.760558125753502,0.760201413762811,0.771767761172035,0.798794133055764,0.857783057479984,0.675533979377439])
    data_ = [coef(i, y=pe) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PE_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pet = np.array([0.272012722877454,0.362898850612845,0.353138413806723,0.341344523410971,0.331421982357141,0.324488094390281,0.320061537427762,0.317274219387557,0.313370875863725,0.308973094617144,0.303336251285241,0.297053014299212,0.289528038733828,0.280793805462979,0.273186645986093,0.268727072038837,0.266988294891645,0.267199403202377,0.267732531055554,0.268161441262858,0.266699096975455,0.265177731492659,0.264181270482422,0.264113262466810,0.265900999542747,0.265519699230076,0.261246750829338,0.259319316872283,0.264712921003348,0.274586838740672,0.286553723182985,0.295411654920427,0.304195663307495,0.312577882977639,0.325596694420256,0.343001600296836,0.370023641425205,0.400528076652043,0.425775660953412,0.440224862466637,0.437749710647781,0.423369765742872,0.408785508600997,0.401146746838619,0.403509422727330,0.413695754594912,0.425396797231603,0.438511559281054,0.450354866969741,0.461577777341471,0.473289712437648,0.484819845379607,0.490671412785548,0.488449153974590,0.476393777335769,0.461643251249526,0.445869072218735,0.432348048242373,0.418213443349046,0.400902126670851,0.382405243875322,0.363001488915613,0.344630261880103,0.323208540312560,0.303325065142703,0.285178895465466,0.269927713665824,0.262588423947658,0.260815843624189,0.261028577402170,0.261608442556523,0.267562500600089,0.278945852385303,0.293936982088889,0.312278787758491,0.331976212024267,0.352617739794657,0.374194793265344,0.393887932570298,0.414545924744799,0.437588952584871,0.468848185315387,0.516738885451100,0.571693995198065,0.634427274902687,0.696070971882516,0.749519738057609,0.795129486102615,0.834543697590993,0.867101756815230,0.896292889420886,0.929258404984734,0.959798293961445,0.991756941633368,1.02738183233191,1.05660773047283,1.07908961270862,1.08531507528262,1.07052036774178,1.03181604496146,0.967114552943948,0.883399336916655,0.794133807714521,0.697447743169723,0.610653114909790,0.534448814287875,0.470740003878753,0.418446528320403,0.373069449852651,0.336327244669907,0.300029790692429,0.268624519100717,0.241933587016132,0.223122281074510,0.212254164313843,0.211259928817015,0.215302746655911,0.221343231163216,0.228198426389516,0.237872019864321,0.256236677310290,0.292525642103885,0.348128562054030,0.416749643911666,0.492337838371771,0.560845197806362,0.609159220492757,0.626599286466856,0.607150534655707,0.551811676160454,0.471945908366300,0.379297755558916,0.293290430008393,0.231813293480824,0.198581560676265,0.185295466097533,0.185450687794481,0.194778864220615,0.208252449017295,0.222424699184626,0.233795269214781,0.241961453001448,0.243403446071623,0.238677965464269,0.228828634646721,0.222207080688684,0.217472698424842,0.214727369759387,0.212373581054222,0.208946708573777,0.206812710346102,0.201222337063961,0.197112956251101,0.198902689934772,0.205793919299678,0.218365476853247,0.236122286064683,0.256522611476142,0.274717598740410,0.292929064745759,0.308986640052716,0.323090683898046,0.336490322915469,0.349880387765552,0.359895337082693,0.364021191850653,0.371581616805955,0.386376940411243,0.418051393129805,0.464141795869782,0.527583608262214,0.603960553292917,0.691253637680648,0.776929028736211,0.866488365913093,0.945152404764694,1.01855331981493,1.09022697640873,1.16754141055074,1.25355091320986,1.35124421724538,1.45651224504790,1.56129652776536,1.66013202765098,1.73423133979863,1.77313990848341,1.76894359273774,1.72107138403223,1.63225845615893,1.51333652773435,1.36885032004303,1.22615221226694,1.10425696274998,1.02244166459680,0.973685904525834,0.941979517743345,0.906099673628226,0.852261584670696,0.778615758716337,0.687292675485545,0.603319741345499,0.529313344908076,0.469451046573507,0.419326885023270,0.374993370395584,0.330766797138072,0.288819299014079,0.248396581866505,0.214247104168250,0.188245581320439,0.172347381788622,0.163040806171760,0.159459695447417,0.161025317267520,0.168676325583308,0.181234132174509,0.195749884656420,0.207008543507150,0.210191600006376,0.206367790291090,0.200981601362366,0.201533237320556,0.207815979262364,0.214727028288850,0.222624549619153,0.232668580887415,0.247297233764891,0.263183439123556,0.281417111734806,0.298911073323481,0.308114632910175,0.304064747954063,0.288142513615475,0.267147866905096,0.247879512483264,0.233510295623439,0.222667926508135,0.210907522374215,0.201333035689554,0.195371320170956,0.193977875564135,0.198549904790742,0.210549184355222,0.223438807423191,0.237309722516247,0.253209663806640,0.276457846824384,0.308213207212797,0.347764141870564,0.392068512545344,0.428474610195628,0.444170541765822,0.433489148342291,0.398222469932266,0.348968021416371,0.299387442806985,0.265004025186602,0.245474632323332,0.234601304509895,0.234314083166786,0.241277739055782,0.256650243755203,0.280688722319311,0.310422072862907,0.337912168792724,0.362617009347716,0.384218022864863,0.401809914454587,0.418489142281082,0.431396076663210,0.441189427383175,0.447456465337255,0.455676975145487,0.461256484619563,0.463202254841243,0.455116218478382,0.440378357365317,0.420771847776988,0.395885380226589,0.369905564875205,0.344922867033398,0.321686054463348,0.300553193138240,0.282338584981687,0.267211318778828,0.255965139658371,0.244483298217803,0.233658417999046,0.223478775423923,0.218516791342784,0.218187031307306,0.218865893575909,0.221276755430487,0.222206109784466,0.223061899699517,0.222742339018101,0.222552150651906,0.227046691628032,0.234236442946591,0.243442146154377,0.251630167299339,0.257554147381402,0.259416645526615,0.256273610617619,0.251302525906309,0.247903744968572,0.248045565471057,0.250002512205685,0.251907189629860,0.251477127736289,0.252188758932070,0.251244765874872,0.247649233332398,0.241839552557005,0.235699390202931,0.230884391577671,0.228585033680674,0.227440759998951,0.226933161193436,0.223953121832736,0.220682690377320,0.220107102342006,0.227190864308438,0.233235790581553,0.241439533557174,0.245957125937824,0.245470559443672,0.242573606927069,0.239352644710686,0.235688101034487,0.234311643199233,0.235561461790730,0.238771790903066,0.243432390137285,0.246173827124570,0.249945072247158,0.253556064448370,0.261327734612423,0.270219002099940,0.282841714528937,0.296207873429230,0.313242140215847,0.327508332031628,0.334775850282387,0.333758760342646,0.329978655548331,0.331612664588903,0.341299232766417,0.357164879045027,0.377809836379274,0.405556505399639,0.445839072491324,0.508697851594103,0.600986035257226,0.733727795819965,0.914495458930015,1.15292954658247,1.49489796758304,2.01315658195505,2.76669367318327,3.74690418006222,4.76530544653771,5.47208736509920,5.47311019296479,5.02722956627457,3.92829095492549,2.70865941347202,1.70247122119030,1.04829338211003,0.695505733821621,0.519479880151060,0.411461316305135,0.336008386341281,0.281236759701160,0.239105966012148,0.206808973658114,0.183272701714151,0.167265320143505,0.156647446931321,0.153713408282484,0.154738044349797,0.156723567241801,0.160182581266526,0.161518456521911,0.162647228132089,0.162124177592661,0.161550204984557,0.159674456370203,0.157793629857961,0.148981689243649,0.159084855405682,0.201225164755696,0.279310911775166,0.250854983145041])
    data_ = [coef(i, y=pet) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PET_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pfa = np.array([0.554907772009540,0.562108368998073,0.570222095562700,0.579248951703424,0.587179800952102,0.599389267157063,0.615854110840355,0.630579898494152,0.642295792888830,0.655301157903246,0.658160333652337,0.655136222012573,0.656733599857081,0.662541344684160,0.668405386218819,0.673908977693139,0.675347048942414,0.674647680993821,0.674532797983417,0.674378065295972,0.678866217724603,0.673997107481293,0.670560941256014,0.679883145836190,0.676618757371975,0.679885287563043,0.680208595369984,0.671891046532404,0.665089911955497,0.669919629742573,0.669739055340429,0.668420747590262,0.666359231715442,0.663925386154782,0.662377706499176,0.663718716337418,0.666370717404852,0.665575929553295,0.671243061315582,0.674653802289991,0.679990261957698,0.682475702053628,0.686041123842744,0.689125389844573,0.691365569967649,0.697289143656402,0.694361891653565,0.688781200619492,0.689817361800986,0.685690012344634,0.676631813240554,0.681473954884419,0.681186653323772,0.681518514687492,0.681951739835158,0.676809919394889,0.666793538255352,0.662010672381016,0.665386895089040,0.666066917661100,0.663874833931025,0.664098858848181,0.661042424250632,0.658814343570153,0.657117221166599,0.647296391624549,0.639196988842407,0.640839123286066,0.643439233793864,0.648041341450252,0.643515418402412,0.638145255229797,0.630146231139741,0.635734573959477,0.638747641086881,0.645274909837375,0.653972431088401,0.653154609812989,0.645310905202101,0.642723580307225,0.637791223715285,0.642956308341703,0.657346208661693,0.661882994794195,0.665667143226606,0.663987972435113,0.659745514466768,0.664382887019216,0.662752089341914,0.667031018155279,0.664142649454463,0.661279834665737,0.659847329441632,0.668775923998337,0.664517844233807,0.670716223258052,0.664192816054119,0.662599095748198,0.652060022131432,0.650874009993510,0.644312096523492,0.641747112571846,0.634816703386610,0.625797702920841,0.616787850959751,0.618484031864754,0.611324022718548,0.608359162109000,0.607371580539202,0.601253221872208,0.596069587245133,0.591743916587564,0.584196050095892,0.585506295240895,0.593610692062652,0.605932645854307,0.606855711156019,0.607161872953716,0.600499823494313,0.599235530355160,0.606862233683928,0.618339540796342,0.622461403257757,0.630382005281451,0.632696228924591,0.641214790984255,0.645084831852060,0.646242268533984,0.642966352349196,0.640885399290048,0.634932570753427,0.644582912268451,0.647345711359750,0.656303039601631,0.669492652420824,0.688912748531814,0.707798846301351,0.737796647942049,0.761785794282184,0.783230928438263,0.814702502638917,0.848977918507831,0.887185963280651,0.924433111148346,0.961425276747934,0.996411826891402,1.03052492585927,1.04544386558772,1.04195897402966,1.01277277321089,0.955301522928134,0.882235390658319,0.795495446336172,0.711748206930112,0.636087275121887,0.574827476838174,0.526582010978608,0.509220303522915,0.501861104649581,0.509159459548888,0.520228163908935,0.527780448374702,0.526108138756984,0.533065105462671,0.529186957892747,0.520849309839500,0.518041457131203,0.516336797255444,0.508732106200612,0.512436303159389,0.522480101551377,0.534958597456694,0.554082950053673,0.578337235277675,0.597560627471675,0.620865289260112,0.643270404449752,0.660466282414595,0.677254815331157,0.697307624079325,0.722333612228946,0.756037654476465,0.790188494113202,0.841237409616040,0.903928675463395,0.979355754394673,1.07022844078446,1.17092642512048,1.26549274035122,1.36488428580774,1.42503823962920,1.45046845056809,1.43489078792394,1.38400131719325,1.30511755638231,1.22277412281686,1.13136596362296,1.05678860784796,0.998860255696200,0.955154056745688,0.925386866850978,0.908874141451442,0.899651560695440,0.900057769741515,0.905073253375517,0.900769262768805,0.900917635227649,0.892897847147362,0.884462291877316,0.877348699657905,0.878509608408482,0.884864718999225,0.899694158029895,0.911846622854891,0.918215418390161,0.928747806191359,0.943461055853059,0.956982644252834,0.966683462587024,0.981183176636125,0.989484162978113,1.00596053520813,1.02604827846325,1.05088209438514,1.07834125175100,1.12225251126448,1.16309571116481,1.20034766804858,1.24707173626316,1.32144985765402,1.42317189620396,1.58128939821159,1.76005187370170,1.92418670655509,1.99038727098272,1.91262048521751,1.69416503530740,1.41003726993728,1.13720939327749,0.938023563713840,0.807704412445812,0.731933115163123,0.686001726676051,0.659362865834773,0.637564792661260,0.619375118290736,0.604156204295838,0.598532578518260,0.594344077547909,0.589202533311504,0.582462638174198,0.572338547508821,0.559491728396257,0.550466208879588,0.548368837627646,0.541597529576447,0.542915327226586,0.542038603319466,0.540657157489754,0.540000577851942,0.544508758362655,0.541312172071980,0.543093916837852,0.546170407209621,0.549875829054751,0.552258312767148,0.552563726645505,0.553486139617195,0.553491065032212,0.557709022712597,0.555925662940155,0.549249226452471,0.550648309814881,0.556654641881480,0.559263128531122,0.565060669506474,0.565654233216522,0.567124141147910,0.571379734012043,0.574976282522339,0.573277684807852,0.570755864694438,0.571547278299736,0.573037530562730,0.576526346772430,0.581694979885206,0.586810104464813,0.590960954730680,0.592838107758406,0.587755573106347,0.591203284012284,0.592343110772799,0.606788493975660,0.618336053399232,0.633774744792069,0.635040626160258,0.642989038478072,0.641487860207124,0.641730190855768,0.646504815917378,0.657316699607249,0.655912454760591,0.660436539831588,0.659401139931787,0.659060417493902,0.670001926932255,0.684736677257764,0.688279630948970,0.687413489652069,0.688428562584552,0.688841832215340,0.696086737968256,0.698783186095126,0.705697269298328,0.704539179714853,0.700696159776930,0.693144406869515,0.694652178445138,0.689787881352298,0.701256579933861,0.705622612288842,0.704108034549592,0.703411914389329,0.704323106647003,0.701867422435910,0.705216107144970,0.702276499172222,0.704360485752437,0.707870647881480,0.708960056442136,0.696074735690222,0.688200553502426,0.681925250543707,0.689123847881111,0.693192950986077,0.703386921162665,0.702854309600299,0.698793140216586,0.700395805538729,0.698816144602748,0.697754265915137,0.702453867224017,0.700192565377285,0.696119677367758,0.697993302157386,0.689455894425785,0.689926126978318,0.686183795590861,0.685166655517202,0.693430491065957,0.691138377278738,0.686568927158404,0.681688555906573,0.677519552965525,0.685770575246167,0.693639321643218,0.688582253650284,0.687913294362044,0.678646322840902,0.672096815650411,0.670721211105792,0.670663935043607,0.672053955880668,0.672276448814483,0.664503649734737,0.654578535427884,0.645483266468871,0.639080799630850,0.631464302388608,0.624867597256412,0.617878699847030,0.611262241065752,0.604655248284444,0.597427615384026,0.591280970482769,0.596730407999503,0.601150242638889,0.608434282533388,0.617957532270774,0.623690808211518,0.630566426119252,0.642471102957700,0.642045059866722,0.647529945936444,0.646813833683130,0.645845859952056,0.639815499874592,0.642561246917737,0.637501892587002,0.640423918816530,0.630814733320985,0.649791582610648,0.682310530185939,0.728371576046853])
    data_ = [coef(i, y=pfa) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PFA_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pp = np.array([0.0417637854477889,0.120024734086829,0.161323531156112,0.246390937503850,0.384462876284972,0.565122202852232,0.765748686898205,0.926053840839443,1.00532679571313,0.977888557677969,0.866063159491508,0.705711405755446,0.560010278842835,0.458926473367196,0.407263151695364,0.400830356005048,0.421175461790880,0.461744838613905,0.545839620015009,0.692354119954252,0.951675917866424,1.38028671028499,1.92434139447815,2.38363799295102,2.54836118885552,2.48190261541058,2.11459415978638,1.54573860022457,1.03624105259096,0.722882089668594,0.587302019187644,0.576031685060063,0.682925437303113,0.901041435888880,1.22616319952112,1.56350947445705,1.78535578561754,1.83367966190596,1.70287761192065,1.41206367908233,1.07556912795159,0.814674773945072,0.654558779171058,0.568588763952420,0.526054099967297,0.503027358891868,0.499167670277405,0.502354167873788,0.513783416599189,0.546669420560363,0.610707466990160,0.722155671641388,0.897773437261535,1.13103836012256,1.38089205154772,1.58237140989564,1.69112867829419,1.69608665189575,1.60516781427636,1.45607075505370,1.27070243219500,1.07767447513900,0.888948578973725,0.715215600295056,0.575102459876763,0.469514571726441,0.401767275156643,0.355605645654132,0.324169136396433,0.303512225230389,0.284716531536706,0.265761534469899,0.251247999744461,0.241352950946423,0.234453894732846,0.233144022636587,0.230312387676150,0.228299134142419,0.229162236078751,0.228239156747194,0.233567000487141,0.244605773416429,0.258052097525682,0.276210012578264,0.304835841444353,0.352543035072069,0.416139701891225,0.488506979369911,0.557494898338403,0.604915415890213,0.611502166432870,0.573897482914751,0.508272612756934,0.433913459744899,0.367415153191776,0.320403436563431,0.290707087214301,0.271013832832209,0.260902713487546,0.256021830924102,0.259520055660214,0.269773854216144,0.284321623718118,0.303800973974651,0.330845744227151,0.374548762073185,0.442067886882661,0.545869756589830,0.697329893783400,0.929607997658102,1.29769283250293,1.83504066284227,2.54486860131303,3.25840826899151,3.75151832213526,3.84770992109694,3.73197415097910,3.32249847485382,2.89007366610370,2.65290293482046,2.58679179952286,2.63319444965669,2.66619717251454,2.58820855511742,2.33945865116286,1.89930221001913,1.39000073206754,0.958870036136489,0.657727413716459,0.469543542337964,0.357870584967101,0.291238013799257,0.253032795457575,0.234699065776969,0.235447576717262,0.250727565819847,0.274888739762936,0.297950783045595,0.319493811749766,0.343900065562274,0.369841432498756,0.404283844263420,0.449607260265969,0.516715769529057,0.626921908527263,0.787759107058658,1.00363587422658,1.25372998904354,1.46842848187028,1.58650897926441,1.56825963116942,1.42223801203999,1.16627134563184,0.881401809174754,0.653204976465649,0.497514189822417,0.401363139719189,0.344431631302350,0.308790615976308,0.285824709002225,0.269110196442911,0.262627408760135,0.266189102525658,0.282349109991526,0.321611292991387,0.381097225359150,0.455698273551376,0.517496758448746,0.539718366141631,0.514582499931697,0.447651745202851,0.365190375900941,0.297283306759753,0.253526416517627,0.231919602609618,0.221275246375992,0.218710697160264,0.221808212046189,0.231625251163191,0.242434812047187,0.252241628695723,0.266403360689569,0.284432805146656,0.309606590827652,0.342828010923472,0.389223901518933,0.453572846045845,0.521690934788613,0.589245350758021,0.641420291301625,0.672079166212924,0.683698436706743,0.687622881459965,0.684523932813895,0.678350319122129,0.658899009696647,0.627997936952132,0.607023825950191,0.616899746069196,0.677765064660371,0.798105810169428,0.980678237435175,1.23102156476162,1.62383203037148,2.22258373151685,2.99191038218974,3.73667944794483,4.07042779732212,4.06106678889905,3.65149882927609,2.87832222680026,2.11876578382757,1.53989367974056,1.19018626086768,0.982997872971681,0.851605587094276,0.778162722170128,0.755078420429010,0.798973305900711,0.919148513798962,1.12305594008090,1.41008633567219,1.67294670668204,1.80410879188936,1.79843335478593,1.64532783636875,1.38202524415899,1.13499928747755,0.967502130208070,0.878292972594582,0.832376331449185,0.796711782774703,0.744638751759009,0.671771274629794,0.579489893980992,0.473837109895481,0.373264391179148,0.291624840667968,0.229628900863119,0.188678795936618,0.161631518936534,0.148441447771939,0.143906026549129,0.141473465011910,0.140320209416720,0.137914007715750,0.136945019536369,0.135114633096663,0.136200727967730,0.136409436966134,0.139966913995298,0.145629221113705,0.155142169410328,0.165118898615826,0.178448448276867,0.194952891764620,0.217794593455990,0.251239340870329,0.306426138192653,0.403056890913535,0.580015914837982,0.886010762696903,1.31057754410653,1.73423908136714,1.97751856464773,2.05035121833460,1.98512784325993,1.86648271645187,1.81421647189172,1.87603088356060,2.06646494200787,2.40637332589586,2.86560424359088,3.43029787671277,4.08150471013695,4.72229418407350,5.20951195376234,5.33850241571420,5.24710688381452,4.72994670492001,3.92069124557038,3.05983504440400,2.29962761463789,1.71481083982759,1.29199420376758,0.994455835072305,0.794513877871721,0.659598924861663,0.559851301694710,0.489321215987134,0.436709212779973,0.400639012616803,0.374586773152439,0.353343426218999,0.333675762136695,0.315972575363103,0.303024340038481,0.294721855480087,0.289121068809630,0.283652050020206,0.279926325333303,0.275034695387499,0.272866452261567,0.268063880321440,0.262366086827081,0.260010546317912,0.257433590920741,0.258279256070184,0.262356933641208,0.265938205701517,0.265546622825139,0.265885767296196,0.266730496051093,0.268016539495624,0.270219211099462,0.272022012297105,0.274948623422711,0.278333024534960,0.280068895221149,0.280718672956335,0.284946226132084,0.290358157010081,0.297770520199420,0.300910910179834,0.302675732898841,0.305342829007898,0.309171560676164,0.311690703011133,0.313757958043526,0.313513479040755,0.311227127026076,0.311834242820913,0.313869969603203,0.316388231764610,0.313973107410741,0.314844813634503,0.316720062638966,0.319338556577527,0.324259406375016,0.330523379989326,0.334449800613723,0.332570728516245,0.332512051030130,0.333415852697085,0.332876277908649,0.333498555886651,0.333037366239990,0.332301190630265,0.334279384843441,0.336291935651053,0.340970138992717,0.343968356028260,0.346337734850565,0.348807174919122,0.348882865265198,0.346670046993329,0.346161140907221,0.345921698316085,0.350347110708280,0.355376868190207,0.359128138947147,0.361110821709145,0.363216684850955,0.362121225592497,0.357720099555756,0.356147242794685,0.358288101916903,0.361026911548607,0.365475752480324,0.365693659135921,0.364147597863697,0.363374250295443,0.362702251176342,0.363922854166974,0.365082086490521,0.362480410980261,0.361609714537843,0.363435368236842,0.368850744401611,0.371723041180668,0.376666204500986,0.379135268017533,0.383940944204689,0.385819331443336,0.384146047362762,0.381028727410942,0.379429607757554,0.381891065824226,0.384418456706999,0.386498554209586,0.387231250448191,0.394808774991781,0.409348741014320,0.320220022464694])
    data_ = [coef(i, y=pp) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PP_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    ps = np.array([0.0137129482514854,0.0273137724646977,0.0224695094818470,0.0190922110958182,0.0173609804061122,0.0177330374450110,0.0199833620245157,0.0224322208689485,0.0242431928354522,0.0244705060982569,0.0222742037797203,0.0185623351103223,0.0146585416584067,0.0109633410438230,0.00944660686067937,0.0108625294353842,0.0162460128186306,0.0245197117543355,0.0345991902947961,0.0424812780646388,0.0466669941913924,0.0478353559984222,0.0490194932530226,0.0551586967073081,0.0699352880410054,0.0960589686406882,0.135634199447646,0.191951562204911,0.276004496989179,0.398196677473494,0.554808139541660,0.731565533756397,0.934999664917246,1.30602619276861,2.19850236945420,3.87601986699621,6.12938204107286,8.24330004245108,8.64447379199095,8.63717542415541,6.67124981308654,4.27695048330328,2.27598821936804,1.06484706416598,0.532100474183917,0.356447618574322,0.332892853307535,0.362089722979401,0.524614399136975,0.840459189888462,1.29820756554074,1.82996058709436,2.28568664461730,2.42712136840956,2.42141020231324,2.09134680506880,1.60271528239672,1.12701985600607,0.754310891347795,0.504191631341745,0.354013445563610,0.268893540953060,0.216468114860766,0.183020413156910,0.164347443237714,0.156978584118612,0.159264887387815,0.170977969382851,0.189691529147068,0.213053658222374,0.238781390716680,0.262242534870915,0.281906189825179,0.291711193593305,0.292147660817548,0.284083166474082,0.271397987635037,0.257171016800776,0.245730409544032,0.236917026567807,0.231098082059033,0.228746964488958,0.230932452371651,0.234748780448933,0.240954268952413,0.247300666887180,0.250611619039238,0.246536408247516,0.232213306597282,0.205575622014154,0.170683266355896,0.133625246132190,0.100659261711816,0.0735720098008893,0.0534052029515958,0.0366405612803738,0.0200925525152230,0.00334831029904497,-0.0117936552190102,-0.0232648339342446,-0.0297663149010459,-0.0315542370042871,-0.0299015626823363,-0.0268581226816406,-0.0228071755590458,-0.0164642078096302,-0.00530305371518557,0.0141493475899997,0.0431200278570089,0.0821200309735985,0.134648117209654,0.209924912816978,0.334897608009507,0.519034864706169,0.740722583310533,0.932636116189433,0.995023092348673,0.983710803317260,0.864300066284631,0.683653960296945,0.524031490484947,0.426889336086065,0.383803872568396,0.377158533626092,0.407924407341184,0.484591904985343,0.638824344948846,0.863971240468051,1.11064791920847,1.30168543156807,1.36344366509871,1.35819870245218,1.28964893131113,1.21700745540723,1.18767486022636,1.19799956417477,1.23898826903983,1.28611281821974,1.31911472700834,1.32487684516244,1.29380751004629,1.22604309725623,1.11336365096972,0.968252090730409,0.810769733817421,0.661107472656794,0.533594316461472,0.437663943769662,0.370536696168515,0.322743729490427,0.288397947958853,0.261765970576033,0.238305152893948,0.215373757218115,0.192336126170541,0.168509295408404,0.145995995215746,0.127012582410990,0.112624357700451,0.102430005918152,0.0960607887650011,0.0912916373919201,0.0881205351151940,0.0852920241982606,0.0831380555546574,0.0809406431185841,0.0781826305554223,0.0742834646777573,0.0700530574082085,0.0656670740044644,0.0620342148629749,0.0598840436281095,0.0596980535708483,0.0606501676659703,0.0615375269876472,0.0621738854049966,0.0631526105748086,0.0645532848364188,0.0662478780276785,0.0683231178348281,0.0705278510779016,0.0730216886644253,0.0765451656449846,0.0822983615446804,0.0901622697310592,0.101188525527255,0.116128823352705,0.135022924162650,0.155990378266403,0.175905393040953,0.191293338468272,0.202746991673245,0.211049272301888,0.217383210366728,0.222878231647979,0.228201133085164,0.235447809079858,0.245306430062859,0.259321919008050,0.277459852658043,0.300649010877352,0.326827329749087,0.354247802430946,0.378644479952591,0.398854127858578,0.412460406685389,0.420339801442942,0.422406109856353,0.417299465834030,0.403331420527074,0.379631724414269,0.349132514103235,0.314454896622341,0.280136185939073,0.250000831771911,0.226646121079302,0.209810752688756,0.197592392855621,0.188256264792209,0.180999316916910,0.176026560581205,0.172891843907100,0.171538266072956,0.172172369712458,0.174142726151691,0.176522405084963,0.178285095166209,0.179561080392715,0.179011152580919,0.176436935993730,0.171331985598408,0.161623083952118,0.148411820960117,0.131634359214978,0.113001978657068,0.0946138482481772,0.0786222212077298,0.0654973928606427,0.0557369900030334,0.0491233401399770,0.0454727560541781,0.0437938585690163,0.0436642574002674,0.0442631021235175,0.0453205035340452,0.0461692696562568,0.0465177348839890,0.0465652033628702,0.0464252387391412,0.0459530392222269,0.0452685827487221,0.0448244625635487,0.0450048823525903,0.0455482320663061,0.0467277469756682,0.0482789345611435,0.0504683112314546,0.0545584853565606,0.0605963421053431,0.0702216222979244,0.0846235096213359,0.106037188981143,0.137187537947783,0.178253827148215,0.227187078763912,0.279855485035582,0.333583320734867,0.388377053251212,0.445717876458104,0.502670506635604,0.553101366892303,0.587489183100311,0.597628229454065,0.576440278720634,0.525685056043880,0.444165881404984,0.348322984418154,0.262436910701458,0.199112506023952,0.158866011439681,0.132823313905699,0.115590016593444,0.103521829358606,0.0939501352971533,0.0869747366802896,0.0807950381865882,0.0758271493570440,0.0724878127245430,0.0704711895543477,0.0688499587512975,0.0685393017288069,0.0699851034941860,0.0743133279675113,0.0820119981395096,0.0920183327066118,0.0997861930112888,0.101679831983477,0.0959187127291078,0.0851213923422216,0.0738389980179583,0.0652631573257375,0.0600696116770916,0.0573210552425743,0.0554254580492480,0.0546393633614451,0.0543025292618894,0.0545873086720998,0.0548448814232228,0.0551882691210148,0.0551901196437275,0.0547831160458510,0.0541745638601509,0.0536466287621013,0.0530277131749306,0.0530918177002094,0.0529025704432170,0.0530460637857274,0.0525662396415655,0.0527630961297869,0.0527641691034655,0.0528808378690823,0.0532743129912895,0.0534920151465752,0.0535755570037099,0.0540252615754173,0.0544371840323766,0.0551638974145338,0.0561873601788604,0.0580874324261250,0.0602706006780364,0.0628707730753627,0.0659364450403582,0.0702576061516683,0.0753932476770401,0.0824288147050778,0.0915830442234454,0.103054410701375,0.117916466073672,0.137423164842288,0.165737163650265,0.206899840279427,0.280636374721670,0.418216266477112,0.615806812353527,0.812184989758028,0.887790235929022,0.881856487193703,0.767546241988305,0.606687207151437,0.528339938533959,0.529684510842887,0.620552707511336,0.864486984065973,1.22259035311465,1.62840462752001,1.94709208078371,1.97622370337246,1.91865875204969,1.57567702824900,1.16013082259446,0.789291954417210,0.521819723705658,0.356724035388521,0.265185991193202,0.212131294003015,0.177638012888481,0.155416779349012,0.138247517402222,0.124320003537466,0.111447504793359,0.100233410732474,0.0903305829192195,0.0813358050899154,0.0740858403570839,0.0680734293580180,0.0626138698505335,0.0584715533471284,0.0552239232551036,0.0529724586786898,0.0515817353423620,0.0509683952608663,0.0504220069421408,0.0503175687064113,0.0501753000933401,0.0497148233722754,0.0491231839581532,0.0483350363443957,0.0477217359827812,0.0478398786277961,0.0488617428439127,0.0492793098013797,0.0473670437106681,0.0427562688210705,0.0210948535361986])
    data_ = [coef(i, y=ps) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PS_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pu = np.array([-0.0526864330637916,-0.0247708894878706,0.000673045822102426,0.0236567834681042,0.0419496855345912,0.0648787061994609,0.0706918238993711,0.0914645103324349,0.0920934411500449,0.100359389038634,0.0938903863432165,0.103144654088050,0.105840071877808,0.124348607367475,0.125965858041330,0.140610961365678,0.138005390835580,0.147619047619048,0.144294699011680,0.142497753818509,0.136657681940701,0.141150044923630,0.138274932614555,0.141509433962264,0.123989218328841,0.107906558849955,0.0830548068283917,0.0766576819407008,0.0681491464510333,0.0706469002695418,0.0575112309074573,0.0542677448337826,0.0461635220125786,0.0447259658580413,0.0415992812219227,0.0481940700808625,0.0492632524707996,0.0653728661275831,0.0787870619946092,0.0866127583108715,0.0902964959568733,0.0994609164420485,0.110512129380054,0.128212039532794,0.134411500449236,0.141778975741240,0.124977538185085,0.105750224618149,0.0986522911051213,0.105929919137466,0.115723270440252,0.133423180592992,0.145193171608266,0.164869721473495,0.185265049415993,0.211500449236298,0.241868823000898,0.266936208445642,0.290745732255166,0.310601976639713,0.330907457322552,0.350673854447439,0.376010781671159,0.394339622641509,0.407277628032345,0.432973944294699,0.471428571428572,0.515633423180593,0.564510332434861,0.596406109613657,0.605660377358491,0.611320754716981,0.579065588499551,0.526594788858940,0.468283917340521,0.410242587601078,0.375112309074573,0.363162623539982,0.361096136567835,0.375651392632525,0.389847259658580,0.400179694519317,0.402156334231806,0.381132075471698,0.355705300988320,0.318149146451033,0.267295597484277,0.216082659478886,0.166397124887691,0.118508535489668,0.0819766397124888,0.0493800539083558,0.0194788858939802,0.00203863432165319,-0.00467565139263253,-0.00804132973944295,-0.00945193171608266,-0.00709164420485175,-0.00428032345013477,0.00525696316262354,0.0175831087151842,0.0349146451033244,0.0477178796046721,0.0660736747529200,0.0865408805031447,0.106828391734052,0.113746630727763,0.121114106019766,0.123899371069182,0.125876010781671,0.116082659478886,0.102066486972147,0.0755435759209344,0.0547079964061096,0.0418688230008985,0.0350494159928122,0.0189398023360288,0.0158221024258760,-0.00105031446540881,-0.00830368373764600,-0.0184276729559748,-0.0159209344115005,-0.0178346810422282,-0.00817250673854448,-0.0117969451931716,-0.00603234501347709,-0.00190386343216532,0.0144564240790656,0.0192902066486972,0.0238364779874214,0.0240341419586703,0.0174123989218329,0.0233423180592992,0.0321743036837377,0.0419766397124888,0.0578975741239892,0.0747888589398023,0.0794339622641510,0.0855345911949686,0.0921832884097035,0.112219227313567,0.124797843665768,0.136388140161725,0.137376460017969,0.120035938903863,0.0955974842767296,0.0746720575022462,0.0597214734950584,0.0441509433962264,0.0402336028751123,0.0145283018867925,-0.0107547169811321,-0.0218688230008985,-0.0108625336927224,-0.00902066486972147,2.42857142857143e-05,-0.000692452830188679,0.00518328840970350,0.0157592093441150,0.0363522012578616,0.0362893081761006,0.0356783468104223,0.0410512129380054,0.0541150044923630,0.0671787960467206,0.0841419586702606,0.0959568733153639,0.120485175202156,0.148966756513926,0.186972147349506,0.236747529200359,0.271428571428571,0.298292902066487,0.313926325247080,0.312848158131177,0.313836477987421,0.341060197663971,0.379155435759209,0.422282120395328,0.461185983827493,0.483647798742138,0.511500449236298,0.532794249775382,0.553009883198563,0.550404312668464,0.545193171608266,0.527583108715184,0.535220125786164,0.540071877807727,0.560646900269542,0.575471698113208,0.595867026055705,0.599910152740341,0.605929919137466,0.604312668463612,0.622731356693621,0.637466307277628,0.672776280323450,0.715274034141959,0.770350404312669,0.828751123090746,0.892093441150045,0.935309973045822,0.971248876909254,0.992812219227314,1,0.982030548068284,0.952380952380952,0.913746630727763,0.862893081761006,0.803593890386343,0.736837376460018,0.656963162623540,0.576010781671159,0.480952380952381,0.389757412398922,0.308445642407907,0.236747529200359,0.168283917340521,0.119317160826595,0.0656783468104223,0.0328212039532794,0.0164779874213837,0.00809164420485175,-0.00349865229110512,-0.0125247079964061,-0.0274573225516622,-0.0419407008086253,-0.0494159928122192,-0.0433872416891285,-0.0391554357592093,-0.0346630727762803,-0.0256424079065589,-0.00662803234501348,0.0296046720575022,0.0868104222821204,0.139802336028751,0.182389937106918,0.199820305480683,0.200988319856244,0.179694519317161,0.161185983827493,0.132794249775382,0.114645103324349,0.101078167115903,0.0993710691823899,0.102425876010782,0.133153638814016,0.152470799640611,0.175741239892183,0.168463611859838,0.155435759209344,0.124887690925427,0.117789757412399,0.124707996406110,0.147978436657682,0.158939802336029,0.177088948787062,0.178346810422282,0.193171608265948,0.206918238993711,0.222461814914645,0.230817610062893,0.230727762803234,0.220485175202156,0.228930817610063,0.239442946990117,0.263791554357592,0.298652291105121,0.326145552560647,0.349505840071878,0.377717879604672,0.406738544474394,0.421563342318059,0.431087151841869,0.426415094339623,0.423000898472597,0.420215633423181,0.412039532794250,0.396495956873315,0.362893081761006,0.318059299191375,0.278975741239892,0.248337825696316,0.226504941599281,0.223899371069182,0.211141060197664,0.189757412398922,0.168733153638814,0.143216531895777,0.115543575920934,0.0946091644204852,0.0698742138364780,0.0446091644204852,0.0244204851752022,0.0150763701707098,0.00451392632524708,0.00402156334231806,0.00244743935309973,-0.00421743036837376,-0.0100359389038634,-0.0183557951482480,-0.0194699011680144,-0.00787331536388140,-0.000324618149146451,0.00461365678346810,0.00892542677448338,0.00133782569631626,0.00608715184186882,0.0121293800539084,0.0151841868823001,0.0240341419586703,0.0285804132973944,0.0393261455525606,0.0415363881401617,0.0448876909254268,0.0488409703504043,0.0603414195867026,0.0686702605570530,0.0844833782569632,0.0851752021563342,0.0924528301886793,0.106379155435759,0.117879604672058,0.128571428571429,0.137196765498652,0.137376460017969,0.139802336028751,0.142677448337826,0.131626235399820,0.113746630727763,0.104222821203953,0.0813027852650494,0.0697394429469901,0.0541509433962264,0.0429110512129380,0.0236927223719677,0.0154896675651393,-0.000772955974842767,-0.00232255166217430,-0.00931716082659479,-0.00644204851752022,-0.0196136567834681,-0.0110062893081761,0.00686253369272237,0.0364869721473495,0.0670440251572327,0.108176100628931,0.135220125786164,0.195238095238095,0.254177897574124,0.318688230008985,0.389038634321653,0.472506738544474,0.565408805031447,0.650673854447439,0.693441150044924,0.693620844564241,0.645193171608266,0.590116801437556,0.552650494159928,0.530907457322552,0.525516621743037,0.544474393530997,0.569092542677448,0.617520215633423,0.665408805031447,0.706469002695418,0.738724168912848,0.743216531895777,0.716172506738545,0.668553459119497,0.593710691823900,0.509074573225517,0.429919137466307,0.360736747529200,0.310332434860737,0.288319856244385,0.272237196765499,0.277178796046721,0.277178796046721,0.269901168014376,0.252201257861635,0.234411500449236,0.199730458221024,0.172776280323450,0.137376460017969,0.101527403414196,0.0637646001796945,0.0439532794249775,0.0444923629829290,0.0185175202156334,-0.0142677448337826,-0.0538814016172507])
    data_ = [coef(i, y=pu) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PU_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    cp = np.array([0.0263930528299482,0.0272417147419536,0.0279461988832696,0.0285065052538962,0.0289663052003762,0.0294387182544484,0.0299839908777260,0.0315158007073951,0.0322603096079217,0.0336225901049128,0.0365387489317339,0.0499926036235572,0.0778517965775309,0.114035793130507,0.148630956930030,0.165340505219841,0.154174878380751,0.124269342809043,0.0878394253933675,0.0576695882153190,0.0397276684127693,0.0317657535952706,0.0277805631564505,0.0263308322568668,0.0260928400659639,0.0268521823065064,0.0255356976138329,0.0241231049683762,0.0220853930160974,0.0202443865113581,0.0202515624898106,0.0210830396937076,0.0208316128328695,0.0214527333951335,0.0224038829455804,0.0233581969856433,0.0228388708452160,0.0257069767395071,0.0301692172417187,0.0391788256810173,0.0496947829161115,0.0580667176345002,0.0593110379347142,0.0570055389005043,0.0488206890882668,0.0399155995966222,0.0322989529299425,0.0278740703414384,0.0268921926387488,0.0286334362533291,0.0297130334039748,0.0295707667529318,0.0294798107761722,0.0303409273193785,0.0301850968490183,0.0306423061890970,0.0318736983680953,0.0319782957180583,0.0332075172106781,0.0330337393483433,0.0294074395794324,0.0269649819754977,0.0234092526866517,0.0215654430531872,0.0199916586533902,0.0189338212545241,0.0176799003547866,0.0179586313642438,0.0165920083658802,0.0173775718167192,0.0168981147821563,0.0172185854101283,0.0181061842714747,0.0186448543243146,0.0185011546557242,0.0184409920969725,0.0187529532594119,0.0206524743691298,0.0310268990249218,0.0495605613450156,0.0730735460689375,0.0900439640922030,0.0929561984889032,0.0781576029861939,0.0558955033068315,0.0361763078045207,0.0252510163082053,0.0219254288740362,0.0259190437667260,0.0354269473940416,0.0559226481992616,0.0841866885417789,0.109732477221044,0.123846666929842,0.119501349763758,0.0997867121829580,0.0741866791568009,0.0519257813500989,0.0362601001176334,0.0306634771589423,0.0316418436298982,0.0372885111313859,0.0438683758099154,0.0488560799224185,0.0544127710521860,0.0596930821711608,0.0702898696284780,0.0955503478695121,0.141147146229073,0.198861422627448,0.251031297062484,0.269647732695358,0.246516300162286,0.189703380077562,0.127778780602976,0.0822216857251451,0.0588046298293268,0.0517008085840010,0.0523372455068533,0.0496314014620922,0.0455106207376441,0.0419359427799404,0.0389331245842857,0.0370511861934069,0.0354436539418459,0.0331968961208077,0.0292084142256076,0.0310116265450131,0.0388845493885750,0.0525623395379813,0.0688800160994842,0.0851316969133207,0.0952693963521725,0.102663575511027,0.109324184201031,0.115438136743607,0.121738944257483,0.125975216852063,0.124637535650731,0.118815211029252,0.113397459960202,0.110127630201902,0.110330265335161,0.114608198810290,0.118712117624409,0.120804103536308,0.123712424858677,0.126057796611241,0.126979556946397,0.122436301558203,0.108543958625649,0.0876592492383620,0.0647060426633620,0.0428750207178345,0.0259800488571694,0.0139227925283225,0.00637923528712983,0.00450385195694429,0.00601052286229449,0.00619915774746047,0.00720942281556150,0.00796886982015602,0.00882457485317288,0.0122455192875875,0.0155144860963310,0.0159965943191400,0.0191247331811979,0.0210852563790445,0.0220817053769514,0.0245016939362633,0.0250187409662964,0.0236273434305450,0.0234849438864864,0.0220853760355828,0.0190780898381940,0.0189988692514627,0.0192793140517879,0.0179382273373854,0.0191628876041077,0.0186184676671881,0.0164782446801197,0.0175184420570319,0.0181079359872499,0.0189196824806392,0.0217614820786856,0.0236976191770170,0.0244778881976563,0.0283148324713722,0.0331644034151606,0.0400853087938369,0.0474926279586209,0.0570635760432450,0.0705356820863339,0.0924085519731307,0.118881531606738,0.142879678031927,0.154792736405356,0.148102887757333,0.127466531586637,0.101540914473617,0.0787417367292479,0.0653972707427286,0.0601889977373373,0.0612305276715636,0.0653027241706470,0.0703020779995001,0.0761017333685723,0.0847724445958751,0.0995468204465137,0.133366439200778,0.194358833187845,0.288784942709834,0.403878058620251,0.505185287393342,0.550662962666629,0.518412503594146,0.418762786391726,0.288735504597393,0.174632585326684,0.0996319624122974,0.0611208081755498,0.0451968703496436,0.0395452875064796,0.0352656910154757,0.0338706208698051,0.0328848697340516,0.0332801458648468,0.0329918714287217,0.0341717431474209,0.0351016413637494,0.0353614903111941,0.0324338390114716,0.0298593591907709,0.0246645037770467,0.0205355838976690,0.0176841927769784,0.0150561590740762,0.0136266969227453,0.0124150386025740,0.00968621577891962,0.00781914258446489,0.00785845622762691,0.00881027773518250,0.0110755984695494,0.0121338078455204,0.0117823394731460,0.0121918757811343,0.0127839395985214,0.0132905270158597,0.0130388805373790,0.0124976257498643,0.0126007422740280,0.0135729548564688,0.0147140517119063,0.0154733617156302,0.0151416474108874,0.0144056832955525,0.0150517812104295,0.0186249076922567,0.0240996831448221,0.0304477050508249,0.0331578287635916,0.0324299894129892,0.0295490007403811,0.0262443624002078,0.0255206707353847,0.0287603327006488,0.0346465364047539,0.0449218890331961,0.0675724336263521,0.109237048412778,0.166494410800796,0.219027821866033,0.243048920883292,0.222839327397035,0.169556656441786,0.109094493501864,0.0611685352849540,0.0330181779514392,0.0224654869106190,0.0193305130448228,0.0177634955003384,0.0173537517614028,0.0171385404801369,0.0167981936758351,0.0167134875862278,0.0167516287925591,0.0176619673691689,0.0177583694029586,0.0179488586696755,0.0173423446340064,0.0132798576721393,0.00973819525059115,0.00699053324540059,0.00511006083875295,0.00583718700063374,0.00692432916513053,0.00660410721331717,0.00744867596429700,0.00897455699252021,0.0101833625162820,0.0113888602832508,0.0132224435553140,0.0159010861726480,0.0196688893869868,0.0238910321905974,0.0266084771704498,0.0308080106794057,0.0410029066253959,0.0589815118887286,0.0958434544818605,0.182390289438239,0.348107807735070,0.589316621439125,0.838911330875011,1,0.998160142755767,0.840470256957102,0.607125939513659,0.391707525755882,0.241442404349401,0.159081442815540,0.118235530978352,0.0935244934399863,0.0761155349117235,0.0635122349052574,0.0533972928735075,0.0448253777644091,0.0393769852317586,0.0349837802220294,0.0318102569478771,0.0288238070020712,0.0238848026208224,0.0190154509623892,0.0154082566722216,0.0111662958501553,0.00900499348587105,0.00904242697859333,0.00889578567601388,0.00946280121265876,0.0111834299040602,0.0122887595504479,0.0135119736632152,0.0152543874486910,0.0154388282285112,0.0168712640134945,0.0196695743713390,0.0237557258044627,0.0295249887603636,0.0369141569231837,0.0443045113583995,0.0540024346403839,0.0622798666127154,0.0662715632404522,0.0640233253259922,0.0580035682636731,0.0516245577214249,0.0471706354067390,0.0464665369327340,0.0468863171283877,0.0440738703624403,0.0397686452093451,0.0351237924920635,0.0296639259863211,0.0250204956018425,0.0217804298027334,0.0189971189616372,0.0172408549326381,0.0164455594022085,0.0157646640992297,0.0148766183737371,0.0141729718421518,0.0136710154977996,0.0144083963814033,0.0155564836956471,0.0161096918842322,0.0161455980438479,0.0158900179395781,0.0155746330969027,0.0155037545475434,0.0153635439791689,0.0155485544696307,0.0156932935251805,0.0166869123976452,0.0182622449587645,0.0184832560670540,0.0177365413067300,0.0170962635999400,0.0169813002023612,0.0167662240858557,0.0177459457322311,0.0186404545733478,0.0197396314531858,0.0210434763717450])
    data_ = [coef(i, y=cp) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'Copper_phthalocyanine_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pmma = np.array([0.246491555156292,0.259547726797899,0.273410040462719,0.288078496150752,0.304538760120715,0.320671224295655,0.335424080649714,0.357375462411372,0.379886230075871,0.409375261579033,0.434844684366993,0.466464713596430,0.502315152977158,0.549568028106290,0.604264496456662,0.666802199415732,0.729123604332209,0.791850619257919,0.845108039901608,0.881682267904007,0.898487697872399,0.892680301100453,0.866471292674109,0.837611724458885,0.814924506314711,0.810924780594249,0.823000462934829,0.842255494978322,0.858371780607926,0.871983193799480,0.876650901400180,0.872537638518143,0.859426821742476,0.838584731043056,0.814961746889868,0.780107980724455,0.736416492795602,0.668722806419966,0.579325202342296,0.486850937749770,0.409131025983517,0.349849067387562,0.318532496064324,0.302945057713199,0.298936386955252,0.303255594877270,0.302596060672485,0.300540627379290,0.293318771576092,0.285401160328085,0.281552917348370,0.283911782765559,0.279021860043151,0.278230497954302,0.274643466620038,0.271952589276563,0.275214737162613,0.280453941042576,0.278593601270101,0.279150685314851,0.279438809028901,0.283637534035012,0.291025252115860,0.302067040240041,0.311921659318195,0.324126629840686,0.339605584559145,0.349868994243515,0.351834734486542,0.357545454981311,0.353374155452780,0.346358811755677,0.341370174691780,0.336397957379288,0.326721964713107,0.322866291647387,0.320785002345497,0.322676377024465,0.326772432992692,0.336530981296015,0.338972745317364,0.343573172577922,0.353433168681749,0.359252431304307,0.367137262928643,0.378327275848673,0.384194356592849,0.398806802571342,0.414300743871409,0.427071959871451,0.440747956731523,0.459048866610503,0.476259210236163,0.497843042974464,0.515701046039793,0.533308061907580,0.548167994433668,0.560369978445207,0.566380797744588,0.556942792378156,0.540399896137141,0.515736357289334,0.487629519828036,0.461257367185547,0.439892805138017,0.421361566172797,0.414731825194970,0.413076777500144,0.415587837796788,0.420120732963142,0.424950639531470,0.431402798890435,0.439815856007193,0.454954056396933,0.471506836912413,0.482476268993399,0.491229094543888,0.494992587270670,0.490531813504385,0.482507152077402,0.473682482463553,0.468552467409378,0.471859705877183,0.478095098782279,0.488030881288666,0.499794063657613,0.513815638433011,0.526327317491762,0.534372087675176,0.535580752111203,0.526474682422705,0.520978586872761,0.508823308272534,0.495716172097481,0.486813868853755,0.484523743566275,0.479387062229303,0.480118477577173,0.476275592020659,0.474927755245220,0.470700761161569,0.463475518182057,0.449968843702309,0.434183679967282,0.419387706703394,0.407850817768326,0.395950005654303,0.389825501521286,0.384238997381518,0.386095215153392,0.387288943879424,0.391354142025425,0.395160396876566,0.399985098306439,0.405460593644951,0.413924348846010,0.415755961672766,0.426201023276701,0.433822429030378,0.433634788399355,0.431507065955868,0.422859738495644,0.404823957932239,0.395441571926757,0.385203999538735,0.371617637443501,0.358076235172734,0.348050978890640,0.338660183505304,0.335229141285852,0.332082621405553,0.330865845531954,0.327694381970108,0.330772223207654,0.329226141578103,0.328436782846589,0.322290484568458,0.319961197925968,0.312370608783826,0.311813805735635,0.306319379176086,0.305951306423185,0.300974841963154,0.305284395988814,0.304226823781986,0.305480240545275,0.301442646194796,0.299950360866039,0.297878954999467,0.303555502600703,0.306215544983574,0.310221163201549,0.311605402915140,0.317593276537626,0.321949275422788,0.326285940204868,0.331236395509649,0.336413229297862,0.342987246437126,0.348770592174300,0.349916633830572,0.356096480994356,0.362103522366761,0.368408099950775,0.373252920086162,0.370617710176461,0.361430794547763,0.356766170002652,0.347110506167415,0.343529201067877,0.339114328274802,0.338895803345973,0.336025102267842,0.336533684723984,0.329395000000192,0.325634788159368,0.318987783313148,0.318129271891722,0.312688732530142,0.314153520312094,0.309865057152629,0.308462874545488,0.306710356641837,0.306639430465143,0.299170476384783,0.299107256654442,0.294488506980874,0.291603501444264,0.289814632264214,0.288798546102543,0.286945165470945,0.287864042239267,0.288159364681739,0.293417833973773,0.303416147360187,0.315603308501307,0.325796844336703,0.330250828126665,0.326276798253175,0.320370361963071,0.312125398540149,0.305297452997182,0.300537661721609,0.296782701787875,0.289346796780355,0.281240163140757,0.276202611088203,0.279380430183739,0.281225739912665,0.285269303677405,0.282576902797704,0.279377466492325,0.284913465228184,0.298814721257483,0.311489136715422,0.332122841947171,0.356259967021240,0.384344649654555,0.422234805431895,0.462069238019744,0.497643873158890,0.533606488990056,0.573975023793447,0.616568484067432,0.669487107654469,0.729436921219172,0.797063206477210,0.861577315745001,0.919195842357968,0.967367378047767,0.993281410942631,1,0.998188419903694,0.978846740799908,0.949898168282410,0.925343817833012,0.893625583210609,0.857463277382599,0.814496056191261,0.767811024705818,0.711929329963253,0.658627237501631,0.609217374878963,0.569431866288898,0.539091102255855,0.521169613929623,0.504991416397228,0.500639505435327,0.498475072131311,0.497074176423186,0.496762108033779,0.484105933764011,0.463120391003717,0.441115032288810,0.414642155259562,0.391227434246467,0.372280296803058,0.353874816588264,0.336462145639105,0.322444539773989,0.316688811214252,0.315015669312669,0.310829847822120,0.307713992335217,0.300357732261141,0.296814522536994,0.300760217706845,0.300638296959505,0.298867285463135,0.297945146287038,0.294491405636892,0.292312571071656,0.292874606159210,0.293743044101099,0.297372235858051,0.301023336003070,0.301313449640233,0.302412979287819,0.301865004260556,0.301426326440529,0.304687172014734,0.308007611528574,0.306785374345498,0.311235410029145,0.313962526244308,0.312847453146536,0.315034487011055,0.317783988438186,0.318017679547658,0.321778303974888,0.322204315278867,0.322610733645184,0.321050798982848,0.321477006488532,0.319019105480681,0.321664332734069,0.320268594171249,0.324459542378788,0.329056696027505,0.333424730501879,0.334777978391311,0.338989481253224,0.337268753276717,0.337781380216490,0.338307804254954,0.339575633052531,0.340042189793867,0.338182168351005,0.337154394117717,0.341165073658239,0.343389224757811,0.349726063481443,0.354349686674770,0.357566659072654,0.356927825539698,0.355943680955973,0.347877160014423,0.343773141741728,0.342838477252423,0.343525230644132,0.341118544038209,0.339281263341865,0.332926167972454,0.330852506010615,0.329716890635764,0.324888456738243,0.322708827455581,0.322634971255099,0.324041118412948,0.325169512988638,0.322465646906040,0.320560741357971,0.320176659659335,0.321994587184649,0.323176761264870,0.325752676535795,0.326816122672627,0.329292617764365,0.326791093545374,0.329745646798247,0.325758907067372,0.328052033521797,0.330995544761210,0.331579554671013,0.330935098126187,0.332909732563761,0.330727364246643,0.330944552112024,0.332291195379761,0.331979760498165,0.331887095416693,0.342906512085459,0.360070376696234,0.383378689249018])
    data_ = [coef(i, y=pmma) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PMMA_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    pvc = np.array([0.000854574115561438,0.00127314667523915,0.00127950964210801,0.00129389548024631,0.00130606811251717,0.00133954285126205,0.00142170811909039,0.00151825922505703,0.00161647023542424,0.00168424966511429,0.00173128028979718,0.00180542268635608,0.00191525302752730,0.00203531944583538,0.00216700519494747,0.00227019591851640,0.00237947295822076,0.00252277803813686,0.00262596876170578,0.00267631919518982,0.00267161613272153,0.00260853976549977,0.00249953937652884,0.00235927745468046,0.00221790892989836,0.00207239064411484,0.00193185207153303,0.00178024746961407,0.00166875722404228,0.00161481033102367,0.00155201061453534,0.00149668046784959,0.00147261185404129,0.00146348237983814,0.00146265242763785,0.00148118802677758,0.00148616773997930,0.00151687597138989,0.00153762477639704,0.00159295492308280,0.00160208439728595,0.00161951339349196,0.00164662516536798,0.00168701617244857,0.00173791990739947,0.00177969416814721,0.00181565876349295,0.00183834412363411,0.00182119177816152,0.00181123235175809,0.00181150900249152,0.00184028067876811,0.00188260824098271,0.00188814125565128,0.00190086718938901,0.00191635963046102,0.00192355254953016,0.00191497637679387,0.00193904499060217,0.00193627848326789,0.00197390298301420,0.00202729657456595,0.00205911140891026,0.00208649983151970,0.00212495428346630,0.00215151275387546,0.00218664739702091,0.00225608673111153,0.00236149066054789,0.00245804176651453,0.00255376292028088,0.00267410598932239,0.00281630446630477,0.00296846236969059,0.00308742218506496,0.00320084898577075,0.00336960593316229,0.00353559637321954,0.00374031791595683,0.00396993802470270,0.00418019258210855,0.00439044713951441,0.00459240217491741,0.00475285960030609,0.00491055051836048,0.00505164239240915,0.00518166823712067,0.00524253139847499,0.00526466345714929,0.00519550077379210,0.00507377445108345,0.00490225099635762,0.00470306246828891,0.00454537155023452,0.00438214761751155,0.00425212177280003,0.00411379640608565,0.00396163850269984,0.00375415045262827,0.00349963177787381,0.00319254946376788,0.00285780207631908,0.00249013325159226,0.00209369275058885,0.00175036919040376,0.00142668783229211,0.00118019202880709,0.000962744552332083,0.000796477461541399,0.000701862910708763,0.000645426161089296,0.000617207786279562,0.000608078312076413,0.000627167212682998,0.000660365300694449,0.000775728656534241,0.000955274982529506,0.00128725586264402,0.00171440459505802,0.00229398788159127,0.00298782792103060,0.00368498776927107,0.00432405096349151,0.00474456007830322,0.00501567779706341,0.00510143952442632,0.00506824143641487,0.00497694669438338,0.00491055051836048,0.00484968735700615,0.00477775816631467,0.00466986438027746,0.00445131030086874,0.00417742607477427,0.00390907486334837,0.00363795714458819,0.00338067196249944,0.00315658486842215,0.00297952839902774,0.00283290351031050,0.00270564417293327,0.00257949143848975,0.00243397315270623,0.00225193697011010,0.00201706049742908,0.00182810804649724,0.00164773176830169,0.00148533778777901,0.00131602753892061,0.00121034695875082,0.00113177815045705,0.00107312819497016,0.00104103670989242,0.00106095556269929,0.00114616398859535,0.00125405777463257,0.00138685012667837,0.00150746984645331,0.00158465540107993,0.00166958717624256,0.00175645550653919,0.00181482881129266,0.00191829618559502,0.00200267465929079,0.00208954298958742,0.00217696462135091,0.00229979754699328,0.00240603142862992,0.00248985660085884,0.00255569947541488,0.00255929593494945,0.00247906722225511,0.00238002625968762,0.00228651831178870,0.00219937333075864,0.00212329437906573,0.00202010365549680,0.00193544853106760,0.00184719694710383,0.00178273732621493,0.00170804162818916,0.00166184095570656,0.00164385865803369,0.00165547798883770,0.00170223196278716,0.00179020689601750,0.00194596125893789,0.00214708634214060,0.00244559248351023,0.00280800494430191,0.00313721931708213,0.00347749971919951,0.00383991217999118,0.00423552272879431,0.00458686916024883,0.00493544908436907,0.00528679551582359,0.00559111132259523,0.00577370080665821,0.00584286349001540,0.00585116301201826,0.00584009698268111,0.00585946253402112,0.00585946253402112,0.00584839650468397,0.00588712760736400,0.00595629029072119,0.00606971709142698,0.00612781374544702,0.00609184915010128,0.00592585871004403,0.00562984242527525,0.00517613522245209,0.00464773232160316,0.00414146147942853,0.00366285571059678,0.00329767674247081,0.00302102600904205,0.00286610159832195,0.00276069766888559,0.00269651469873012,0.00262624541243921,0.00254988981001287,0.00245582856064710,0.00240686138083021,0.00231418338513157,0.00220573629762750,0.00207349724704855,0.00193849168913532,0.00178688708721636,0.00165132822783627,0.00154232783886533,0.00141949491322296,0.00132847682192490,0.00126042074150143,0.00121975308368740,0.00116746609506936,0.00107340484570359,0.000998985798411250,0.000869513255166591,0.000783751527803675,0.000715695447380201,0.000680284153501320,0.000665898315363024,0.000672814583698743,0.000702692862909049,0.000781261671202817,0.000917650482783195,0.00108115106623959,0.00132958342485862,0.00167512019091114,0.00216036557734518,0.00273220264334243,0.00339727100650517,0.00416912655277141,0.00509314000242346,0.00636020036152718,0.00795924160074541,0.00952231824461790,0.0107451144863730,0.0112845834165591,0.0113565126072506,0.0111517910645133,0.0108391757357388,0.0104435651869357,0.00978513644137522,0.00870619858100306,0.00716802050313916,0.00558557830792665,0.00424935526546574,0.00328107769846509,0.00257838483555604,0.00214210662893888,0.00186213608670898,0.00165077492636941,0.00148450783557872,0.00139376639501409,0.00132764686972462,0.00127369997670601,0.00122749930422341,0.00120536724554910,0.00117355241120480,0.00118129863174080,0.00118489509127538,0.00117991537807366,0.00119623777134596,0.00121864648075369,0.00123330896962541,0.00124658820482999,0.00122694600275655,0.00122860590715712,0.00125903748783428,0.00129389548024631,0.00134175605712948,0.00136361146507036,0.00139846945748238,0.00140012936188295,0.00141202534342039,0.00143747721089583,0.00148063472531072,0.00149170075464787,0.00149419061124873,0.00149280735758159,0.00149446726198216,0.00149031750098073,0.00152351558899218,0.00157856908494450,0.00164441195950055,0.00166267090790684,0.00167290698504371,0.00167373693724399,0.00165713789323827,0.00163417588236368,0.00162891951842854,0.00163777234189826,0.00166488411377427,0.00168480296658114,0.00172740717952917,0.00174262296986776,0.00171966095899317,0.00168895272758258,0.00167207703284342,0.00170444516865459,0.00178578048428264,0.00184581369343668,0.00188731130345100,0.00190031388792215,0.00190557025185729,0.00188205493951585,0.00182257503182867,0.00180597598782294,0.00182119177816152,0.00182534153916295,0.00182810804649724,0.00180542268635608,0.00177250124907806,0.00176475502854206,0.00177111799541092,0.00177416115347863,0.00179186680041808,0.00179103684821779,0.00178218402474807,0.00179408000628550,0.00179131349895122,0.00179103684821779,0.00176641493294263,0.00178439723061550,0.00179159014968465,0.00180763589222351,0.00179297340335179,0.00176530833000891,0.00175313569773805,0.00176918144027692,0.00181150900249152,0.00183834412363411,0.00182036182596124,0.00180680594002323,0.00175285904700462,0.00172934373466318,0.00173487674933175,0.00174262296986776,0.00170499847012144,0.00168701617244857,0.00167207703284342,0.00163666573896454,0.00159461482748337,0.00156528984973992,0.00155284056673563,0.00152960190512761,0.00145767271443613,0.00140261921848381,0.00137052773340607,0.00136748457533836,0.00136914447973893,0.00136693127387150,0.00140040601261638,0.00140151261555010,0.00140068266334981,0.00143360410062783,0.00147759156724300,0.00153568822126304,0.00160263769875280,0.00165547798883770,0.00169559234518487,0.00170029540765316,0.00172353406926117,0.00173460009859832,0.00175313569773805,0.00177139464614435,0.00174760268306947,0.00185521981837326,0.00214846959580775,0.00267272273565525,0.00250534904193085])
    data_ = [coef(i, y=pvc) for i in data]
    im = image(data_,xy)
    plt.savefig(my_path + 'PVC_RSI.png',bbox_inches='tight', transparent=True, dpi=800, pad_inches=0)
    plt.close()
    return