"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const defaults = require("@aws-solutions-constructs/core");
const assertions_1 = require("aws-cdk-lib/assertions");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain'
    };
    return new lib_1.DynamoDBStreamsToLambdaToElasticSearchAndKibana(stack, 'test-dynamodb-stream-lambda-elasticsearch-stack', props);
}
function getDefaultTestLambdaProps() {
    return {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
    };
}
test('check domain names', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain",
        UserPoolId: {
            Ref: "testdynamodbstreamlambdaelasticsearchstackLambdaToElasticSearchCognitoUserPoolF99F93E5"
        }
    });
    template.hasResourceProperties('AWS::Elasticsearch::Domain', {
        DomainName: "test-domain",
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.lambdaFunction !== null);
    expect(construct.dynamoTable !== null);
    expect(construct.elasticsearchDomain !== null);
    expect(construct.elasticsearchRole !== null);
    expect(construct.identityPool !== null);
    expect(construct.userPool !== null);
    expect(construct.userPoolClient !== null);
    expect(construct.cloudwatchAlarms !== null);
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        domainName: 'test-domain'
    };
    try {
        new lib_1.DynamoDBStreamsToLambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('Test minimal deployment with VPC construct props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const construct = new lib_1.DynamoDBStreamsToLambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        deployVpc: true,
        vpcProps: {
            vpcName: "vpc-props-test"
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "vpc-props-test"
            }
        ]
    });
    template.hasResourceProperties("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "testlambdaelasticsearchkibanaVpcisolatedSubnet1Subnet70A13487"
                },
                {
                    Ref: "testlambdaelasticsearchkibanaVpcisolatedSubnet2Subnet26B35F4A"
                },
                {
                    Ref: "testlambdaelasticsearchkibanaVpcisolatedSubnet3SubnetB4A5AAE6"
                }
            ]
        }
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with an existing private VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = new ec2.Vpc(stack, 'existing-private-vpc-test', {
        natGateways: 1,
        subnetConfiguration: [
            {
                cidrMask: 24,
                name: 'application',
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
            },
            {
                cidrMask: 24,
                name: "public",
                subnetType: ec2.SubnetType.PUBLIC,
            }
        ]
    });
    const construct = new lib_1.DynamoDBStreamsToLambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        existingVpc: vpc
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "Default/existing-private-vpc-test"
            }
        ]
    });
    template.hasResourceProperties("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "existingprivatevpctestapplicationSubnet1Subnet1F7744F0"
                },
                {
                    Ref: "existingprivatevpctestapplicationSubnet2SubnetF7B713AD"
                },
                {
                    Ref: "existingprivatevpctestapplicationSubnet3SubnetA519E038"
                }
            ]
        }
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with an existing isolated VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.getTestVpc(stack, false, {
        vpcName: "existing-isolated-vpc-test"
    });
    const construct = new lib_1.DynamoDBStreamsToLambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        existingVpc: vpc
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "existing-isolated-vpc-test"
            }
        ]
    });
    template.hasResourceProperties("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
//# sourceMappingURL=data:application/json;base64,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