"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsEventSource = void 0;
/**
 * Use an Amazon SQS queue as an event source for AWS Lambda.
 */
class SqsEventSource {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10)) {
            throw new Error(`Maximum batch size must be between 1 and 10 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`SqsEventSource:${this.queue.node.uniqueId}`, {
            batchSize: this.props.batchSize,
            eventSourceArn: this.queue.queueArn,
        });
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.queue.grantConsumeMessages(target);
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('SqsEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.SqsEventSource = SqsEventSource;
//# sourceMappingURL=data:application/json;base64,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