"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cache = exports.LocalCacheMode = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Local cache modes to enable for the CodeBuild Project.
 */
var LocalCacheMode;
(function (LocalCacheMode) {
    LocalCacheMode["SOURCE"] = "LOCAL_SOURCE_CACHE";
    LocalCacheMode["DOCKER_LAYER"] = "LOCAL_DOCKER_LAYER_CACHE";
    LocalCacheMode["CUSTOM"] = "LOCAL_CUSTOM_CACHE";
})(LocalCacheMode = exports.LocalCacheMode || (exports.LocalCacheMode = {}));
/**
 * Cache options for CodeBuild Project.
 *
 * A cache can store reusable pieces of your build environment and use them across multiple builds.
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-caching.html
 */
class Cache {
    /**
     *
     */
    static none() {
        return { _toCloudFormation: () => undefined, _bind: () => { return; } };
    }
    /**
     * Create a local caching strategy.
     *
     * @param modes the mode(s) to enable for local caching.
     */
    static local(...modes) {
        return {
            _toCloudFormation: () => ({
                type: 'LOCAL',
                modes,
            }),
            _bind: () => { return; },
        };
    }
    /**
     * Create an S3 caching strategy.
     *
     * @param bucket the S3 bucket to use for caching.
     * @param options additional options to pass to the S3 caching.
     */
    static bucket(bucket, options) {
        return {
            _toCloudFormation: () => ({
                type: 'S3',
                location: core_1.Fn.join('/', [bucket.bucketName, options && options.prefix || core_1.Aws.NO_VALUE]),
            }),
            _bind: (project) => {
                bucket.grantReadWrite(project);
            },
        };
    }
}
exports.Cache = Cache;
//# sourceMappingURL=data:application/json;base64,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