"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectNotificationEvents = exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const notifications = require("@aws-cdk/aws-codestarnotifications");
const ec2 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_canned_metrics_generated_1 = require("./codebuild-canned-metrics.generated");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
const VPC_POLICY_SYM = Symbol.for('@aws-cdk/aws-codebuild.roleVpcPolicy');
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     *
     * Will fail if this Project does not have a VPC set.
     *
     * @stability stable
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    enableBatchBuilds() {
        return undefined;
    }
    /**
     * Add a permission only if there's a policy attached.
     *
     * @param statement The permissions statement to add.
     * @stability stable
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state changes.
     *
     * You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *     const rule = project.onStateChange('OnBuildStarted', { target });
     *     rule.addEventPattern({
     *       detail: {
     *         'build-status': [
     *           "IN_PROGRESS",
     *           "SUCCEEDED",
     *           "FAILED",
     *           "STOPPED"
     *         ]
     *       }
     *     });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     * @stability stable
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @stability stable
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @stability stable
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @stability stable
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @param metricName The name of the metric.
     * @param props Customization properties.
     * @returns a CloudWatch metric associated with this build project.
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensions: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.buildsSum, props);
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     * @stability stable
     */
    metricDuration(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.durationAverage, props);
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricSucceededBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.succeededBuildsSum, props);
    }
    /**
     * Measures the number of builds that failed because of client error or because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    metricFailedBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.failedBuildsSum, props);
    }
    /**
     * Defines a CodeStar Notification rule triggered when the project events emitted by you specified, it very similar to `onEvent` API.
     *
     * You can also use the methods `notifyOnBuildSucceeded` and
     * `notifyOnBuildFailed` to define rules for these specific event emitted.
     *
     * @stability stable
     */
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    /**
     * Defines a CodeStar notification rule which triggers when a build completes successfully.
     *
     * @stability stable
     */
    notifyOnBuildSucceeded(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [ProjectNotificationEvents.BUILD_SUCCEEDED],
        });
    }
    /**
     * Defines a CodeStar notification rule which triggers when a build fails.
     *
     * @stability stable
     */
    notifyOnBuildFailed(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [ProjectNotificationEvents.BUILD_FAILED],
        });
    }
    /**
     * Returns a source configuration for notification rule.
     *
     * @stability stable
     */
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.projectArn,
        };
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ProjectName: this.projectName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * A representation of a CodeBuild Project.
 *
 * @stability stable
 */
class Project extends ProjectBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props, environmentVariables),
            fileSystemLocations: core_1.Lazy.any({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            // The 'alias/aws/s3' default is necessary because leaving the `encryptionKey` field
            // empty will not remove existing encryptionKeys during an update (ref. t/D17810523)
            encryptionKey: core_1.Lazy.string({ produce: () => this._encryptionKey ? this._encryptionKey.keyArn : 'alias/aws/s3' }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            queuedTimeoutInMinutes: props.queuedTimeout && props.queuedTimeout.toMinutes(),
            concurrentBuildLimit: props.concurrentBuildLimit,
            secondarySources: core_1.Lazy.any({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.any({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.any({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
            logsConfig: this.renderLoggingConfiguration(props.logging),
            buildBatchConfig: core_1.Lazy.any({
                produce: () => {
                    const config = this._batchServiceRole ? {
                        serviceRole: this._batchServiceRole.roleArn,
                    } : undefined;
                    return config;
                },
            }),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                    'codebuild:BatchPutCodeCoverages',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        const bindFunction = this.buildImage.bind;
        if (bindFunction) {
            bindFunction.call(this.buildImage, this, this, {});
        }
    }
    /**
     * @stability stable
     */
    static fromProjectArn(scope, id, projectArn) {
        const parsedArn = core_1.Stack.of(scope).parseArn(projectArn);
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i, {
                    account: parsedArn.account,
                    region: parsedArn.region,
                });
                this.projectArn = projectArn;
                this.projectName = parsedArn.resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK, or in a different CDK Stack (and exported using the {@link export} method).
     *
     * @param scope the parent Construct for this Construct.
     * @param id the logical name of this Construct.
     * @param projectName the name of the project to import.
     * @returns a reference to the existing Project
     * @stability stable
     * @note if you're importing a CodeBuild Project for use
     * in a CodePipeline, make sure the existing Project
     * has permissions to access the S3 Bucket of that Pipeline -
     * otherwise, builds in that Pipeline will always fail.
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable}, which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty}, which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables.
     * @param validateNoPlainTextSecrets whether to throw an exception if any of the plain text environment variables contain secrets, defaults to 'false'.
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     * @stability stable
     */
    static serializeEnvVariables(environmentVariables, validateNoPlainTextSecrets = false, principal) {
        var _d;
        const ret = new Array();
        const ssmIamResources = new Array();
        const secretsManagerIamResources = new Set();
        const kmsIamResources = new Set();
        for (const [name, envVariable] of Object.entries(environmentVariables)) {
            const envVariableValue = (_d = envVariable.value) === null || _d === void 0 ? void 0 : _d.toString();
            const cfnEnvVariable = {
                name,
                type: envVariable.type || BuildEnvironmentVariableType.PLAINTEXT,
                value: envVariableValue,
            };
            ret.push(cfnEnvVariable);
            // validate that the plain-text environment variables don't contain any secrets in them
            if (validateNoPlainTextSecrets && cfnEnvVariable.type === BuildEnvironmentVariableType.PLAINTEXT) {
                const fragments = core_1.Tokenization.reverseString(cfnEnvVariable.value);
                for (const token of fragments.tokens) {
                    if (token instanceof core_1.SecretValue) {
                        throw new Error(`Plaintext environment variable '${name}' contains a secret value! ` +
                            'This means the value of this variable will be visible in plain text in the AWS Console. ' +
                            "Please consider using CodeBuild's SecretsManager environment variables feature instead. " +
                            "If you'd like to continue with having this secret in the plaintext environment variables, " +
                            'please set the checkSecretsInPlainTextEnvVariables property to false');
                    }
                }
            }
            if (principal) {
                const stack = core_1.Stack.of(principal);
                // save the SSM env variables
                if (envVariable.type === BuildEnvironmentVariableType.PARAMETER_STORE) {
                    ssmIamResources.push(stack.formatArn({
                        service: 'ssm',
                        resource: 'parameter',
                        // If the parameter name starts with / the resource name is not separated with a double '/'
                        // arn:aws:ssm:region:1111111111:parameter/PARAM_NAME
                        resourceName: envVariableValue.startsWith('/')
                            ? envVariableValue.substr(1)
                            : envVariableValue,
                    }));
                }
                // save SecretsManager env variables
                if (envVariable.type === BuildEnvironmentVariableType.SECRETS_MANAGER) {
                    // We have 3 basic cases here of what envVariableValue can be:
                    // 1. A string that starts with 'arn:' (and might contain Token fragments).
                    // 2. A Token.
                    // 3. A simple value, like 'secret-id'.
                    if (envVariableValue.startsWith('arn:')) {
                        const parsedArn = stack.parseArn(envVariableValue, ':');
                        if (!parsedArn.resourceName) {
                            throw new Error('SecretManager ARN is missing the name of the secret: ' + envVariableValue);
                        }
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretName = parsedArn.resourceName.split(':')[0];
                        secretsManagerIamResources.add(stack.formatArn({
                            service: 'secretsmanager',
                            resource: 'secret',
                            // since we don't know whether the ARN was full, or partial
                            // (CodeBuild supports both),
                            // stick a "*" at the end, which makes it work for both
                            resourceName: `${secretName}*`,
                            sep: ':',
                            partition: parsedArn.partition,
                            account: parsedArn.account,
                            region: parsedArn.region,
                        }));
                        // if secret comes from another account, SecretsManager will need to access
                        // KMS on the other account as well to be able to get the secret
                        if (parsedArn.account && core_1.Token.compareStrings(parsedArn.account, stack.account) === core_1.TokenComparison.DIFFERENT) {
                            kmsIamResources.add(stack.formatArn({
                                service: 'kms',
                                resource: 'key',
                                // We do not know the ID of the key, but since this is a cross-account access,
                                // the key policies have to allow this access, so a wildcard is safe here
                                resourceName: '*',
                                sep: '/',
                                partition: parsedArn.partition,
                                account: parsedArn.account,
                                region: parsedArn.region,
                            }));
                        }
                    }
                    else if (core_1.Token.isUnresolved(envVariableValue)) {
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        let secretArn = envVariableValue.split(':')[0];
                        // parse the Token, and see if it represents a single resource
                        // (we will assume it's a Secret from SecretsManager)
                        const fragments = core_1.Tokenization.reverseString(envVariableValue);
                        if (fragments.tokens.length === 1) {
                            const resolvable = fragments.tokens[0];
                            if (core_1.Reference.isReference(resolvable)) {
                                // check the Stack the resource owning the reference belongs to
                                const resourceStack = core_1.Stack.of(resolvable.target);
                                if (core_1.Token.compareStrings(stack.account, resourceStack.account) === core_1.TokenComparison.DIFFERENT) {
                                    // since this is a cross-account access,
                                    // add the appropriate KMS permissions
                                    kmsIamResources.add(stack.formatArn({
                                        service: 'kms',
                                        resource: 'key',
                                        // We do not know the ID of the key, but since this is a cross-account access,
                                        // the key policies have to allow this access, so a wildcard is safe here
                                        resourceName: '*',
                                        sep: '/',
                                        partition: resourceStack.partition,
                                        account: resourceStack.account,
                                        region: resourceStack.region,
                                    }));
                                    // Work around a bug in SecretsManager -
                                    // when the access is cross-environment,
                                    // Secret.secretArn returns a partial ARN!
                                    // So add a "*" at the end, so that the permissions work
                                    secretArn = `${secretArn}-??????`;
                                }
                            }
                        }
                        // if we are passed a Token, we should assume it's the ARN of the Secret
                        // (as the name would not work anyway, because it would be the full name, which CodeBuild does not support)
                        secretsManagerIamResources.add(secretArn);
                    }
                    else {
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretName = envVariableValue.split(':')[0];
                        secretsManagerIamResources.add(stack.formatArn({
                            service: 'secretsmanager',
                            resource: 'secret',
                            resourceName: `${secretName}-??????`,
                            sep: ':',
                        }));
                    }
                }
            }
        }
        if (ssmIamResources.length !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['ssm:GetParameters'],
                resources: ssmIamResources,
            }));
        }
        if (secretsManagerIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['secretsmanager:GetSecretValue'],
                resources: Array.from(secretsManagerIamResources),
            }));
        }
        if (kmsIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['kms:Decrypt'],
                resources: Array.from(kmsIamResources),
            }));
        }
        return ret;
    }
    /**
     * Enable batch builds.
     *
     * Returns an object contining the batch service role if batch builds
     * could be enabled.
     *
     * @stability stable
     */
    enableBatchBuilds() {
        if (!this._batchServiceRole) {
            this._batchServiceRole = new iam.Role(this, 'BatchServiceRole', {
                assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
            });
            this._batchServiceRole.addToPrincipalPolicy(new iam.PolicyStatement({
                resources: [core_1.Lazy.string({
                        produce: () => this.projectArn,
                    })],
                actions: [
                    'codebuild:StartBuild',
                    'codebuild:StopBuild',
                    'codebuild:RetryBuild',
                ],
            }));
        }
        return {
            role: this._batchServiceRole,
        };
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @stability stable
     */
    addSecondarySource(secondarySource) {
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add.
     * @stability stable
     */
    addFileSystemLocation(fileSystemLocation) {
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     * @stability stable
     */
    addSecondaryArtifact(secondaryArtifact) {
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in.
     * @param options additional options for the binding.
     * @stability stable
     */
    bindToCodePipeline(_scope, options) {
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     * @override true
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            sep: ':',
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(props, projectVars = {}) {
        var _d, _e, _f, _g, _h;
        const env = (_d = props.environment) !== null && _d !== void 0 ? _d : {};
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? ImagePullPrincipalType.CODEBUILD
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_e = this.buildImage.secretsManagerCredentials) === null || _e === void 0 ? void 0 : _e.grantRead(this);
        }
        const secret = this.buildImage.secretsManagerCredentials;
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: secret
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    // Secrets must be referenced by either the full ARN (with SecretsManager suffix), or by name.
                    // "Partial" ARNs (without the suffix) will fail a validation regex at deploy-time.
                    credential: (_f = secret.secretFullArn) !== null && _f !== void 0 ? _f : secret.secretName,
                }
                : undefined,
            certificate: (_g = env.certificate) === null || _g === void 0 ? void 0 : _g.bucket.arnForObjects(env.certificate.objectKey),
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars
                ? Project.serializeEnvVariables(vars, (_h = props.checkSecretsInPlainTextEnvVariables) !== null && _h !== void 0 ? _h : true, this)
                : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    renderLoggingConfiguration(props) {
        var _d, _e, _f, _g, _h;
        if (props === undefined) {
            return undefined;
        }
        let s3Config = undefined;
        let cloudwatchConfig = undefined;
        if (props.s3) {
            const s3Logs = props.s3;
            s3Config = {
                status: ((_d = s3Logs.enabled) !== null && _d !== void 0 ? _d : true) ? 'ENABLED' : 'DISABLED',
                location: `${s3Logs.bucket.bucketName}` + (s3Logs.prefix ? `/${s3Logs.prefix}` : ''),
                encryptionDisabled: s3Logs.encrypted,
            };
            (_e = s3Logs.bucket) === null || _e === void 0 ? void 0 : _e.grantWrite(this);
        }
        if (props.cloudWatch) {
            const cloudWatchLogs = props.cloudWatch;
            const status = ((_f = cloudWatchLogs.enabled) !== null && _f !== void 0 ? _f : true) ? 'ENABLED' : 'DISABLED';
            if (status === 'ENABLED' && !(cloudWatchLogs.logGroup)) {
                throw new Error('Specifying a LogGroup is required if CloudWatch logging for CodeBuild is enabled');
            }
            (_g = cloudWatchLogs.logGroup) === null || _g === void 0 ? void 0 : _g.grantWrite(this);
            cloudwatchConfig = {
                status,
                groupName: (_h = cloudWatchLogs.logGroup) === null || _h === void 0 ? void 0 : _h.logGroupName,
                streamName: cloudWatchLogs.prefix,
            };
        }
        return {
            s3Logs: s3Config,
            cloudWatchLogs: cloudwatchConfig,
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        // If the same Role is used for multiple Projects, always creating a new `iam.Policy`
        // will attach the same policy multiple times, probably exceeding the maximum size of the
        // Role policy. Make sure we only do it once for the same role.
        //
        // This deduplication could be a feature of the Role itself, but that feels risky and
        // is hard to implement (what with Tokens and all). Safer to fix it locally for now.
        let policy = this.role[VPC_POLICY_SYM];
        if (!policy) {
            policy = new iam.Policy(this, 'PolicyDocument', {
                statements: [
                    new iam.PolicyStatement({
                        resources: ['*'],
                        actions: [
                            'ec2:CreateNetworkInterface',
                            'ec2:DescribeNetworkInterfaces',
                            'ec2:DeleteNetworkInterface',
                            'ec2:DescribeSubnets',
                            'ec2:DescribeSecurityGroups',
                            'ec2:DescribeDhcpOptions',
                            'ec2:DescribeVpcs',
                        ],
                    }),
                ],
            });
            this.role.attachInlinePolicy(policy);
            this.role[VPC_POLICY_SYM] = policy;
        }
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "@aws-cdk/aws-codebuild.Project", version: "1.125.0" };
/**
 * Build machine compute type.
 *
 * @stability stable
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 *
 * @stability stable
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
class ArmBuildImage {
    constructor(imageId) {
        this.type = 'ARM_CONTAINER';
        this.defaultComputeType = ComputeType.LARGE;
        this.imagePullPrincipalType = ImagePullPrincipalType.CODEBUILD;
        this.imageId = imageId;
    }
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== ComputeType.LARGE) {
            ret.push(`ARM images only support ComputeType '${ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
/**
 * A CodeBuild image running Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @stability stable
 */
class LinuxBuildImage {
    constructor(props) {
        /**
         * The type of build environment.
         *
         * @stability stable
         */
        this.type = 'LINUX_CONTAINER';
        /**
         * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @stability stable
         */
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Linux build image from a Docker Hub image.
     * @stability stable
     */
    static fromDockerRegistry(name, options = {}) {
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @stability stable
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Linux build image.
     *
     * @stability stable
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @param id The image identifier.
     * @returns A Docker image provided by CodeBuild.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @stability stable
     * @example
     *
     * 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @stability stable
     */
    validate(_) {
        return [];
    }
    /**
     * Make a buildspec to run the indicated script.
     *
     * @stability stable
     */
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
_b = JSII_RTTI_SYMBOL_1;
LinuxBuildImage[_b] = { fqn: "@aws-cdk/aws-codebuild.LinuxBuildImage", version: "1.125.0" };
/**
 * @stability stable
 */
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
/**
 * @stability stable
 */
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
/**
 * @stability stable
 */
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/**
 * The `aws/codebuild/standard:4.0` build image.
 *
 * @stability stable
 */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
/**
 * The `aws/codebuild/standard:5.0` build image.
 *
 * @stability stable
 */
LinuxBuildImage.STANDARD_5_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:5.0');
/**
 * @stability stable
 */
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
/**
 * @stability stable
 */
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/**
 * The Amazon Linux 2 x86_64 standard image, version `3.0`.
 *
 * @stability stable
 */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
/**
 * @stability stable
 */
LinuxBuildImage.AMAZON_LINUX_2_ARM = new ArmBuildImage('aws/codebuild/amazonlinux2-aarch64-standard:1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/**
 * @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section
 */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * Environment type for Windows Docker images.
 *
 * @stability stable
 */
var WindowsImageType;
(function (WindowsImageType) {
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 * @stability stable
 */
class WindowsBuildImage {
    constructor(props) {
        var _d;
        /**
         * The default {@link ComputeType} to use with this image, if one was not specified in {@link BuildEnvironment#computeType} explicitly.
         *
         * @stability stable
         */
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_d = props.imageType) !== null && _d !== void 0 ? _d : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     * @stability stable
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @param repository The ECR repository.
     * @param tag Image tag (default "latest").
     * @returns A Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     * @stability stable
     */
    static fromEcrRepository(repository, tag = 'latest', imageType = WindowsImageType.STANDARD) {
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     *
     * @stability stable
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    /**
     * Allows the image a chance to validate whether the passed configuration is correct.
     *
     * @stability stable
     */
    validate(buildEnvironment) {
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    /**
     * Make a buildspec to run the indicated script.
     *
     * @stability stable
     */
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
_c = JSII_RTTI_SYMBOL_1;
WindowsBuildImage[_c] = { fqn: "@aws-cdk/aws-codebuild.WindowsBuildImage", version: "1.125.0" };
/**
 * (deprecated) Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is based off Windows Server Core 2016.
 *
 * @stability stable
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is based off Windows Server Core 2019.
 *
 * @stability stable
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
/**
 * @stability stable
 */
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
/**
 * The list of event types for AWS Codebuild.
 *
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-buildproject
 * @stability stable
 */
var ProjectNotificationEvents;
(function (ProjectNotificationEvents) {
    ProjectNotificationEvents["BUILD_FAILED"] = "codebuild-project-build-state-failed";
    ProjectNotificationEvents["BUILD_SUCCEEDED"] = "codebuild-project-build-state-succeeded";
    ProjectNotificationEvents["BUILD_IN_PROGRESS"] = "codebuild-project-build-state-in-progress";
    ProjectNotificationEvents["BUILD_STOPPED"] = "codebuild-project-build-state-stopped";
    ProjectNotificationEvents["BUILD_PHASE_FAILED"] = "codebuild-project-build-phase-failure";
    ProjectNotificationEvents["BUILD_PHASE_SUCCEEDED"] = "codebuild-project-build-phase-success";
})(ProjectNotificationEvents = exports.ProjectNotificationEvents || (exports.ProjectNotificationEvents = {}));
//# sourceMappingURL=data:application/json;base64,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