"""The stack module provides support for stacks. The main class in this module
is the Stack class. The StackState acts as a support class to Stack class. For
this reason the Stack class can directly be imported from the ARgorithmToolkit
library without having to import from the stack module:

    >>> st = ARgorithmToolkit.stack.Stack(name="st",algo=algo)
    >>> st = ARgorithmToolkit.Stack(name="st",algo=algo)
"""

from ARgorithmToolkit.utils import State, StateSet, ARgorithmError, ARgorithmStructure
from ARgorithmToolkit.encoders import serialize

class StackState():
    """This class is used to generate states for various actions performed on
    the ``ARgorithmToolkit.stack.Stack`` object.

    Attributes:

        name (str) : Name of the variable for whom we are generating states
        _id (str) : id of the variable for whom we are generating states
    """

    def __init__(self,name):
        self.name = name

    def stack_declare(self,comments=""):
        """Generates the `stack_declare` state when an instance of stack is
        created.

        Args:
            comments (str, optional): Comments for descriptive purpose. Defaults to "".

        Returns:
            State: Returns the `stack_declare` state for respective stack
        """
        state_type = "stack_declare"
        state_def = {
            "variable_name" : self.name,
            "body" : []
        }
        return State(
            state_type=state_type,
            state_def=state_def,
            comments=comments
        )

    def stack_push(self,body,element,comments=""):
        """Generates the `stack_push` state when an element is added to stack.

        Args:
            body (list): contents of stack
            element : Element pushed to stack top
            comments (str, optional): Comments for descriptive purpose. Defaults to "".

        Returns:
            State: Returns the `stack_push` state for respective stack
        """
        state_type = "stack_push"
        state_def = {
            "variable_name" : self.name,
            "body" : list(body),
            "element" : element
        }
        return State(
            state_type=state_type,
            state_def=state_def,
            comments=comments
        )

    def stack_pop(self,body,comments=""):
        """Generates the `stack_pop` state when an element is popped from
        stack.

        Args:
            body (list): contents of stack
            comments (str, optional): Comments for descriptive purpose. Defaults to "".

        Returns:
            State: Returns the `stack_pop` state for respective stack
        """
        state_type = "stack_pop"
        state_def = {
            "variable_name" : self.name,
            "body" : list(body),
        }
        return State(
            state_type=state_type,
            state_def=state_def,
            comments=comments
        )

    def stack_top(self,body,comments=""):
        """Generates the `stack_push` state when top of stack is accessed.

        Args:
            body (list): contents of stack
            comments (str, optional): Comments for descriptive purpose. Defaults to "".

        Returns:
            State: Returns the `stack_top` state for respective stack
        """
        state_type = "stack_top"
        state_def = {
            "variable_name" : self.name,
            "body" : list(body),
        }
        return State(
            state_type=state_type,
            state_def=state_def,
            comments=comments
        )

@serialize
class Stack(ARgorithmStructure):
    """The Stack class is a container interface for the stack, a linear
    container implementation of LIFO.

    Attributes:
        name (str): name given to the rendered block in augmented reality. Essential. Should not be altered after initialisation
        algo (ARgorithmToolkit.utils.StateSet): The stateset that will store the states generated by the instance of Stack Class
        comments (str, optional): Comments for descriptive purpose. Defaults to "".

    Raises:
        ARgorithmError: raised if name is not given or Stateset if not provided

    Example:
        >>> algo = ARgorithmToolkit.StateSet()
        >>> st = ARgorithmToolkit.stack.Stack(name="st",algo=algo)
    """
    def __init__(self,name:str,algo:StateSet,comments=""):
        try:
            assert isinstance(name,str)
            self.state_generator = StackState(name)
        except AssertionError as e:
            raise ARgorithmError('Give valid name to data structure') from e
        try:
            assert isinstance(algo,StateSet)
            self.algo = algo
        except AssertionError as e:
            raise ARgorithmError("Stack structure needs a reference of template to store states") from e
        self.body = []
        state = self.state_generator.stack_declare(comments)
        self.algo.add_state(state)

    def __len__(self):
        """Operator overload for len() function , returns size of stack.

        Returns:
            int: size of stack

        Example:
            >>> len(st)
            0
        """
        return len(self.body)

    def empty(self):
        """Checks whether stack is empty or not.

        Returns:
            bool: if true then stack is empty

        Example:
            >>> st.empty()
            True
            >>> st.push(1)
            >>> st.empty()
            False
        """
        return len(self)==0

    def push(self,element,comments=""):
        """Pushes element to stack top.

        Args:
            element: Element to be pushed to stack top
            comments (str, optional): Comments for descriptive purpose. Defaults to "".

        Example:
            >>> st.push(4)
        """
        self.body.append(element)
        state = self.state_generator.stack_push(self.body,element,comments)
        self.algo.add_state(state)

    def pop(self,comments=""):
        """Pops element from stack top.

        Args:
            comments (str, optional): Comments for descriptive purpose. Defaults to "".

        Raises:
            ARgorithmError: Raised if stack is empty

        Returns:
            element: Element popped from stack top

        Example:
            >>> st.push(3)
            >>> st.push(2)
            >>> st.push(1)
            >>> st.pop()
            1
        """
        if self.empty():
            raise ARgorithmError('Stack is empty')
        item = self.body[-1]
        self.body.pop()
        state = self.state_generator.stack_pop(self.body,comments)
        self.algo.add_state(state)
        return item

    def top(self,comments=""):
        """returns element from stack top.

        Args:
            comments (str, optional): Comments for descriptive purpose. Defaults to "".

        Raises:
            ARgorithmError: Raised if stack is empty

        Returns:
            element: Element at stack top

        Example:
            >>> st.push(1)
            >>> st.push(2)
            >>> st.push(3)
            >>> st.top()
            3
        """
        if self.empty():
            raise ARgorithmError('Stack is empty')
        item = self.body[-1]
        state = self.state_generator.stack_top(self.body,comments)
        self.algo.add_state(state)
        return item

    def __str__(self):
        """String conversion for Stack.

        Returns:
            str: String describing Stack
        """
        return f"Stack({self.body.__str__()})"

    def __repr__(self):
        """Return representation for shell outputs.

        Returns:
            str: shell representation for stack
        """
        return f"Stack({self.body.__repr__()})"
