"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: 'us-east-1' },
});
stack.templateOptions.description = 'Integration Test for public HTTP API with a existing function and ALB';
const myVpc = defaults.buildVpc(stack, {
    defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    constructVpcProps: {
        enableDnsHostnames: true,
        enableDnsSupport: true,
        cidr: '172.168.0.0/16',
    }
});
const testSg = new aws_ec2_1.SecurityGroup(stack, 'lambda-sg', { vpc: myVpc, allowAllOutbound: false });
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        vpc: myVpc,
        securityGroups: [testSg]
    }
});
const loadBalancer = defaults.ObtainAlb(stack, 'existing-alb', myVpc, false, undefined, {
    internetFacing: false,
    vpc: myVpc
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: 'HTTP'
    },
    publicApi: true
};
const albToLambda = new lib_1.AlbToLambda(stack, 'test-one', props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    { id: 'W56', reason: 'All integration tests must be HTTP because of certificate limitations.' },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0].node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
    { id: 'W2', reason: 'Rule does not apply for ELB.' },
    { id: 'W9', reason: 'Rule does not apply for ELB.' }
]);
defaults.addCfnSuppressRules(testSg, [
    { id: 'W29', reason: 'CDK created rule that blocks all traffic.' },
]);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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