"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const fs = require("fs-extra");
const path = require("path");
const lib_1 = require("../lib");
const builder_1 = require("../lib/builder");
jest.mock('../lib/builder', () => {
    return {
        Builder: jest.fn().mockImplementation((options) => {
            return {
                build: jest.fn(() => {
                    require('fs-extra').ensureDirSync(options.outDir); // eslint-disable-line @typescript-eslint/no-require-imports
                })
            };
        })
    };
});
let stack;
const buildDir = path.join(__dirname, '.build');
beforeEach(() => {
    stack = new core_1.Stack();
    fs.removeSync(buildDir);
});
afterEach(() => {
    fs.removeSync(buildDir);
});
test('NodejsFunction with .ts handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1');
    expect(builder_1.Builder).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler1.ts'),
        global: 'handler',
        outDir: expect.stringContaining(buildDir)
    }));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
    });
});
test('NodejsFunction with .js handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler2');
    // THEN
    expect(builder_1.Builder).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler2.js'),
    }));
});
test('throws when entry is not js/ts', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'handler.py'
    })).toThrow(/Only JavaScript or TypeScript entry files are supported/);
});
test('throws when entry does not exist', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'notfound.ts'
    })).toThrow(/Cannot find entry file at/);
});
test('throws when entry cannot be automatically found', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn')).toThrow(/Cannot find entry file./);
});
test('throws with the wrong runtime family', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'handler1', {
        runtime: aws_lambda_1.Runtime.PYTHON_3_8
    })).toThrow(/Only `NODEJS` runtimes are supported/);
});
//# sourceMappingURL=data:application/json;base64,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