"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToDynamoDB = void 0;
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const core_2 = require("@aws-solutions-constructs/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let partitionKeyName;
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            const dynamoTableProps = core_2.overrideProps(defaults.DefaultTableProps, props.dynamoTableProps);
            partitionKeyName = dynamoTableProps.partitionKey.name;
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', dynamoTableProps);
        }
        else {
            partitionKeyName = defaults.DefaultTableProps.partitionKey.name;
            this.dynamoTable = new dynamodb.Table(this, 'DynamoTable', defaults.DefaultTableProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + partitionKeyName + "}");
        // Setup API Gateway Method
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:PutItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "PutItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate
            });
        }
        // Read
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            const getRequestTemplate = "{\r\n\"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n \"KeyConditionExpression\": \"" + partitionKeyName + " = :v1\",\r\n    \"ExpressionAttributeValues\": {\r\n        \":v1\": {\r\n            \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n        }\r\n    }\r\n}";
            this.addActionToPolicy("dynamodb:Query");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "Query",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: apiGatewayResource,
                requestTemplate: getRequestTemplate
            });
        }
        // Update
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:UpdateItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "UpdateItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "PUT",
                apiResource: apiGatewayResource,
                requestTemplate: updateRequestTemplate
            });
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "{\r\n  \"TableName\": \"" + this.dynamoTable.tableName + "\",\r\n  \"Key\": {\r\n    \"" + partitionKeyName + "\": {\r\n      \"S\": \"$input.params('" + partitionKeyName + "')\"\r\n    }\r\n  },\r\n  \"ConditionExpression\": \"attribute_not_exists(Replies)\",\r\n  \"ReturnValues\": \"ALL_OLD\"\r\n}";
            this.addActionToPolicy("dynamodb:DeleteItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "DeleteItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.dynamoTable.tableArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
//# sourceMappingURL=data:application/json;base64,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