"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayLambdaApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const authorizers_1 = require("./authorizers");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const utils_1 = require("./spec/utils");
/**
 * A construct for creating an api gateway api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayLambdaApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, spec, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKey = `${inputSpecAsset.s3ObjectKey}.prepared-spec.json`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(preparedSpecOutputKey),
                            ],
                        }),
                    ],
                }),
            },
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                    },
                ],
            },
        ], true);
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../../lib/construct/prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
            {
                id: "AwsSolutions-L1",
                reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
            },
        ], true);
        const prepareSpecOptions = {
            integrations: Object.fromEntries(Object.entries(integrations).map(([operation, integration]) => [
                operation,
                {
                    functionInvocationUri: utils_1.functionInvocationUri(this, integration.function),
                    methodAuthorizer: api_gateway_auth_1.serializeAsAuthorizerReference(integration.authorizer || defaultAuthorizer || authorizers_1.Authorizers.none()),
                },
            ])),
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(this, integrations, defaultAuthorizer),
            corsOptions: corsOptions && {
                allowHeaders: corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS,
                allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
                allowOrigins: corsOptions.allowOrigins,
                statusCode: corsOptions.statusCode || 204,
            },
            operationLookup,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        prepare_spec_1.prepareApiSpec(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKey,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, preparedSpecOutputKey),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // Grant API Gateway permission to invoke each lambda which implements an integration or custom authorizer
        api_gateway_integrations_1.getLabelledFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-IAM4",
                reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                appliesTo: [
                    {
                        regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                    },
                ],
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-APIG2",
                reason: "This construct implements fine grained validation via OpenApi.",
            },
        ], true);
    }
}
exports.OpenApiGatewayLambdaApi = OpenApiGatewayLambdaApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayLambdaApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayLambdaApi", version: "0.7.1" };
//# sourceMappingURL=data:application/json;base64,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