import pytest

from retro_data_structures.base_resource import Dependency
from retro_data_structures.construct_extensions.json import convert_to_raw_python
from retro_data_structures.formats.pak import PAK, CompressedPakResource, PAKNoData, Pak, PakBody, PakFile
from retro_data_structures.game_check import Game


@pytest.fixture(name="compressed_resource")
def _compressed_resource():
    return {
        "compressed": 1,
        "asset": {"type": "TXTR", "id": 1114209557},
        "contents": {
            "data": b"\x00\x00\n\xcc\x01\xec\x12\x00a\x00@G\x00\x00\x00\x05j\x01\n\xaa\x02\x00\n\xcc\x00\x00-\x0c\x00m"
                    b"\x02\xa0|\x03Z\x03\xaa\xcc-\x00\x00m\x02\x00\xc1\x00\xaaU\x04\xcc-\x0f\x00\xaa\xa0\x00 *\x01\x00"
                    b"\nN\r\n\xcc-\x0c\x00(\xfc\x01>}\x02\xaa\x04#H\x02+\x0c\x00B\x06\xa0\n \x02\r\x00\x00 \x9e\x01\x00"
                    b"\n \x06\x9c\x03L@ 6\xad\x03\x0cN5\x0f\xff6\x0c\x00\x04\xcc\xcc0\x00\xff\xff@7\x0c\x00 (\xfe\x03"
                    b"\xaa\xaa V.\x07\x0f\xff:\x0c\x00:\xec\x03 #\xfc\x03 a\x00\x00 \x91\xfc\x03lE \r\xff\x0fDD\x00T"
                    b"\x154\x0e\x00\x0f\xff\x06\x07O\xff8\x00\x00~\x1ctD6t\x00\xb4\x0bm\x04\xf03\x0c\x00 \x1f|\x03 !"
                    b"\xfe\x13\x88\x88<t\x00 \x1f}\x00\x80 C\xfc\x130\xfe\x1b\xaa\xaa \t|\x19 z\xfd\x1b\n\xcc\xdd/"
                    b"\xfc\x1b\xde\xa1Ww\x01\x00|@\x02\x8d\x00\xc9M\x01\xc7\x0c\x04*\r\x00uM\x04\xc7\xcd\x00u4q\x02WM"
                    b"\x00|\xcd\x00W.p\x00Y\x0buI\x0b\xc7o\x00ww\x9c-v\x01|\xc7v\x00Wu;\xf1\x03\xaeM\x00\xbf6\x0e\x00"
                    b"\xee Ul08\x0c\x00\xbe\x0cWu \x0f\xc4\x05d\x06d\x07/\xcf\x0133\xcfAo\xff\xec\x00/\xcf\x01C3 U\x02"
                    b"\xb0\xec\x00>\x84\x01.\xbc\x05\x8c0\xdc1n3\x00U\x00\x00*\xd4\x00.<\x06Y\x03@<<\x06,\x84\x01l0(<"
                    b"\x06\x04wu\x8a\xaa\x80\x00\xabL\x00pEl\x00Qq\rY\x00\x0e'\x0f\x00\x08\xaa\xa8M\x00\xba\xda\x03"
                    b"\xaa\xf2H\x04M\x00\x00\xec\x00(\xaf\x00\x0c\xcf\x88L\x9d\x01\xaa\x00\x04D`\x05l\nl\x0b\xfe\x04"
                    b"\xfc\xc6M#\xf8\x0e\x0eDA\xc8\rl\nO\x0b\xaa`\x06`Q\x04\n`\t\xb0\x06\x00\nX%L\x00\x05`\xaf\xfb"
                    b"\x06\xa0\x12\"\nn\x03\x82(j\x0c'\x826\x14\x05\x11\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\xff\xff",
            "value": (
                b"\x00\x00\x00\x00\x00@\x00@\x00\x00\x00\x05\x00\x00\x00\x00\n\xaa\xaa\xaa\n\xcc\xcc\xcc"
                b"\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xaa\n\xcc\xcc\xa0\x00\x00\x00\x00"
                b"\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00"
                b"\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc"
                b"\x00\x00\n\xcc\x00\x00\n\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc"
                b"\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\xaa\xaa\xaa\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xaa\xcc\xcc\xa0"
                b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                b"\n\xcc\xcc\xa0\n\xaa\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0c\xcc"
                b"\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff"
                b"\x00\x00\x0f\xff\xcc\xcc0\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00"
                b"\xff\xff@\x00\xff\xff@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xaa\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff"
                b"\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\xff\xff@\x00\xff\xff@\x00"
                b"\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0f\xff"
                b"\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff\x00\x00\x0f\xff"
                b"\x00\x00\x0f\xff\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00\xff\xff@\x00"
                b"\xff\xff@\x00\xff\xff@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\n\xaa\xaa\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00DD\x00\x00\xff\xff\x00\x00\xff\xff\x00\x00\xff\xff\x00\x00\xff\xff"
                b"\x00\x00\xff\xff\x00\x00\xff\xff\x00\x00\x0f\xffDDO\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff@\x00\xff\xfftD"
                b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                b"\x00\x00\x00\x00DD@\x00\xff\xff\xf0\x00\xff\xff\xf0\x00\xff\xff\xf0\x00\xff\xff\xf0\x00"
                b"\xff\xff\xf0\x00\xff\xff\xf0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\xaa\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x80\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xaa"
                b"\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xaa\xaa\xaa\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc"
                b"\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00"
                b"\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc"
                b"\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa"
                b"\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00"
                b"\n\xcc\xcc\xa0\xaa\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0"
                b"\xaa\xaa\xaa\xa0\x00\x00\x00\x00Wwww|\xcc\xcc\xcc|\xcc\xcc\xcc|\xc9ww|\xc7\x00\x00|\xc7\x00\x00"
                b"|\xc7\x00\x00|\xc7\x00\x00u\x00\x00\x00\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00W\x00\x00\x00|"
                b"\x00\x00\x00|\x00\x00\x00W\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00wwwu"
                b"\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7ww\x9c\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7"
                b"|\xc7\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xae\x00\x00\x00\xbf\x00\x00\x00\xbf\x00\x00\x00\xbf"
                b"\x00\x00\x00\xbf\x00\x00\x00\xbf\x00\x00\x00\xbf\x00\x00\x00\xbf\xee \x00\x00\xff0\x00\x00"
                b"\xff0\x00\x00\xff0\x00\x00\xff0\x00\x00\xff0\x00\x00\xff0\x00\x00\xff0\x00\x00\x00\x00|\xc7"
                b"\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00Wu\x00\x00|\xc7\x00\x00\x00\x00\x00\xbf\x00\x00\x00\xbf\x00\x00\x00\xbf"
                b"\x00\x00\x00\xbf\x0033\xcf\x00\xff\xff\xff\x00\xff\xff\xff\x00\xff\xff\xff\xff0\x00\x00"
                b"\xff0\x00\x00\xff0\x00\x00\xff0\x00\x00\xffC3 \xff\xff\xff\xb0\xff\xff\xff\xb0\xff\xff\xff\xb0"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00Wu\x00\x00|\xc7|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc9ww|\xcc\xcc\xcc"
                b"|\xcc\xcc\xccWwww\x00UUU\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00u\x00\x00\x00"
                b"\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00UUU@\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00W\x00\x00\x00|\x00\x00\x00|\x00\x00\x00W\x00\x00|\xc7\x00\x00|\xc7"
                b"\x00\x00|\xc7\x00\x00|\xc7ww\x9c\xc7\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7wwwu\x8a\xaa\x80\x00"
                b"\xab\xaa\x80\x00\xaa\x00\x00\x00\xaa\x00\x00\x00\x88\x00\x00\r\x00\x00\x00\x0e\x00\x00\x00\x0e"
                b"\x00\x00\x00\x0e\x00\x08\xaa\xa8\x00\x08\xaa\xba\x00\x00\x00\xaa\x00\x00\x00\xaa\xf2\x00\x00\x88"
                b"\xf2\x00\x00\x00\xf2\x00\x00\x00\xf2\x00\x00\x00\x00\x00\x00\x0e\x00\x00\x00\x0e\x00\x00\x0c\xcf"
                b"\x88\x00\x0f\xff\xaa\x00\x04D\xaa\x00\x00\x00\xab\xaa\x80\x00\x8a\xaa\x80\x00\xf2\x00\x00\x00"
                b"\xf2\x00\x00\x00\xfc\xc6\x00\x00\xff\xf8\x00\x88DA\x00\xaa\x00\x00\x00\xaa\x00\x08\xaa\xba"
                b"\x00\x08\xaa\xa8\xaa`\x06\xaa\xa0\x00\x00\n`\t\xb0\x06\x00\n\xb0\x00\x00\n\xb0\x00`\xaf\xfb\x06"
                b"\xa0\x12\"\n\xaa`\x06\xaa\x82(\x00\x00\x00\x00'\x82\x00\x00\x00\x00\x00\x00\x00\x00"
                b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
            ),
            "offset1": 13152,
            "offset2": 13664,
            "length": 512,
        },
        "size": 512,
    }


def test_echoes_resource_decode(compressed_resource):
    decoded = CompressedPakResource.parse(compressed_resource["contents"]["data"], target_game=Game.ECHOES)

    assert len(decoded) == len(compressed_resource["contents"]["value"])
    assert decoded == compressed_resource["contents"]["value"]


def test_echoes_resource_encode_decode(compressed_resource):
    raw = compressed_resource["contents"]["value"]
    decoded = CompressedPakResource.build(raw, target_game=Game.ECHOES)
    encoded = CompressedPakResource.parse(decoded, target_game=Game.ECHOES)
    assert raw == encoded


def test_identical_when_keep_data(prime2_paks_path):
    input_path = prime2_paks_path.joinpath("GGuiSys.pak")
    game = Game.ECHOES

    raw = input_path.read_bytes()
    decoded = Pak.parse(raw, target_game=game)
    encoded = decoded.build()

    assert raw == encoded


def test_compare_header_keep_data(prime2_paks_path):
    input_path = prime2_paks_path.joinpath("GGuiSys.pak")
    game = Game.ECHOES

    raw = input_path.read_bytes()
    raw_header = PAKNoData.parse(raw, target_game=game)
    raw_sizes = [(r.compressed, r.offset, r.size) for r in raw_header.resources]

    decoded = PAK.parse(raw, target_game=game)
    # for r in decoded.resources:
    #     r.contents.pop("data")

    encoded = PAK.build(decoded, target_game=game)

    custom_header = PAKNoData.parse(encoded, target_game=game)

    custom_sizes = [(r.compressed, r.offset, r.size) for r in custom_header.resources]
    assert custom_sizes == raw_sizes


def test_compare_from_build():
    game = Game.ECHOES

    source = PakBody(
        named_resources={
            "TXTR_ElevatorIcon_1": Dependency("TXTR", 201335801),
            "TXTR_ElevatorIcon": Dependency("TXTR", 239414538),
            "TXTR_QuaterCurve": Dependency("TXTR", 564256465),
            "TXTR_SaveStationIcon_1": Dependency("TXTR", 568030977),
        },
        files=[
            PakFile(
                asset_id=201335801,
                asset_type="TXTR",
                should_compress=True,
                uncompressed_data=None,
                compressed_data=(
                    b"\x00\x00\x00\x00\x00@\x00@\x00\x00\x00\x05\x00\x00\x00\x00\n\xaa\xaa\xaa\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xaa\n\xcc\xcc\xa0"
                    b"\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xa0\x00\x00"
                    b"\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xaa\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xa0\xcc\xcc\xcc\xa0"
                    b"\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xaa\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xaa\xaa\xa0"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\t\xcc\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff"
                    b"\x00\x00\x0c\xff\xcc\xcc\xcc\xcc\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xfc\x00\x00\xcc\xcc\xcc\xcc"
                    b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\xcc\xcc`\x00\xff\xff\x80\x00\xff\xff\x80\x00"
                    b"\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xaa\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff"
                    b"\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xfc\x00\x00"
                    b"\xff\xfe\x88\x88\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x88\x80\xff\xff\xff\xf0"
                    b"\xff\xff\xff\xf0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xfe\x88\x88\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xff\xff\xf0\xff\xff\xff\xf0\xff\xff\xff\xf0"
                    b"\xff\xff\xff\xf0\x88\x88\x88\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\xaa\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff"
                    b"\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\xff\xfc\x00\x00\xff\xfdDD\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\x00\x00\x00\x00DDDD\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00DD \x00"
                    b"\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\xaa\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x06\x88\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x88\x88@\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xaa\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc"
                    b"\n\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\xaa\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xaa"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\n\xcc\xcc\xa0"
                    b"\xaa\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xaa\xaa\xaa\xa0\x00\x00\x00\x00Wwww|\xcc\xcc\xcc|\xcc\xcc\xcc|\xc9ww|\xc7\x00\x00"
                    b"|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00u\x00\x00\x00\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00W\x00\x00\x00|\x00\x00\x00|\x00\x00\x00W\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00wwwu\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7ww\x9c\xc7"
                    b"\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7|\xc7\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00~\xee\xee\x00\x8f\xff\xff\x00\x8f\xff\xff\x00\x8f\xff\xbb\x00\x8f\xfe\x00\x00\x8f\xfe\x00\x00\x8f\xffU\x00\x8f\xff\xff\xee\xee\xeeP\xff\xff\xffP"
                    b"\xff\xff\xffP\xbb\xbb\xbb@\x00\x00\x00\x00\x00\x00\x00\x00UT\x00\x00\xff\xfb\x00\x00\x00\x00|\xc7\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00Wu\x00\x00"
                    b"|\xc7\x00\x00\x00\x8f\xff\xff\x00\x8f\xff\xff\x00\x8f\xffU\x00\x8f\xfe\x00\x00\x8f\xfe3\x00\x8f\xff\xff\x00\x8f\xff\xff\x00\x8f\xff\xff\xff\xfb\x00\x00"
                    b"\xff\xfb\x00\x00UT\x00\x00\x00\x00\x00\x00333\x10\xff\xff\xffP\xff\xff\xffP\xff\xff\xffP\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00Wu\x00\x00|\xc7|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc9ww|\xcc\xcc\xcc|\xcc\xcc\xccWwww\x005UU"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00u\x00\x00\x00\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00UUU \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00W\x00\x00\x00|\x00\x00\x00|\x00\x00\x00W\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7ww\x9c\xc7\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7wwwu"
                    b"\x8a\xaa\x80\x00\xab\xaa\x80\x00\xaa\x00\x00\x00\xaa\x00\x00\x00\x88\x00\x0c\xff\x00\x00\r\xfe\x00\x00\r\xf0\x00\x00\r\xfc\x00\x08\xaa\xa8\x00\x08\xaa\xba"
                    b"\x00\x00\x00\xaa\x00\x00\x00\xaa\xff\xf3\x00\x88\xee\xe3\x00\x00\x00\x00\x00\x00\xca\x00\x00\x00\x00\x00\r\xff\x00\x00\r\xf4\x00\x00\r\xfc\x88\x00\r\xff"
                    b"\xaa\x00\x03D\xaa\x00\x00\x00\xab\xaa\x80\x00\x8a\xaa\x80\x00\xfe\x00\x00\x00C\x00\x00\x00\xcc\xc2\x00\x00\xff\xf4\x00\x88DA\x00\xaa\x00\x00\x00\xaa\x00\x08\xaa\xba"
                    b'\x00\x08\xaa\xa8\xaa`\x06\xaa\xa0\x00\x00\n`\x9f\xeb\x06\x00\x9c\x80\x00\x00\x9e\xb0\x00`\x9f\xea\x06\xa0\x12"\n\xaa`\x06\xaa\x82(\x00\x00\x00\x00,\xa2'
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                )
            ),
            PakFile(
                asset_id=239414538,
                asset_type="TXTR",
                should_compress=True,
                uncompressed_data=None,
                compressed_data=(
                    b"\x00\x00\x00\x02\x00 \x00 \x00\x00\x00\x04\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\xff"
                    b"\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xff\xff\xff\xff\xff\xff\x0f\x0f\x0f\x0f"
                    b"\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xff\xff\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f"
                    b"\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0f\xff"
                    b"\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xffPPPPPPPPPPPPPPPP\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xffPPPPPPPPPPPPPPPP\xff\xff\xff\xff\xff\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\xff"
                    b"\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPPPPPPPPPPPP\xff\xff\xff\xff\xff\xffPP\xff\xff\xff\xff\xff\xffPP\xff\xff"
                    b"\xff\xff\xff\xffPPPPPPPP\xff\xff\xff\xff\xffPPP\xff\xff\xff\xff\xffPPP\xff\xff\xff\xff\xffPPPPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff"
                    b"\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPPPP\xff\xff\xffPPP"
                    b"PP\xff\xff\xffPPPPP\xff\xff\xffPPPPP\xff\xff\xffPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0f"
                    b"PP\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPPPP\xff\xff\xff\xff\xff\xff"
                    b"PP\xff\xff\xff\xff\xff\xffPP\xff\xff\xff\xff\xff\xffPP\xff\xff\xffPPP\xff\xff\xffPPPPP\xff\xff\xffPPPPP\xff\xff\xffPPPPPPPPPPPPPPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff"
                    b"\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff"
                    b"\xff\xffPPPP\xff\xff\xffPPPPP\xff\xff\xff\xff\xff\xffPP\xff\xff\xff\xff\xff\xffPP\xff\xff\xff\xff\xff\xffPPPPPPPP\xff\xff\xff\xff\xffPPP\xff\xff\xff\xff\xffPPP"
                    b"\xff\xff\xff\xff\xffPPPPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xffPP"
                    b"\x0f\x0f\x0f\xff\xff\xffPP\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0f\xff\xff\xff\xff\xffPPPPPPPPPPPPPPPP\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xffPPPPPPPPPPPPPPPP\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xffPP\xff\xff\xff\x0f\x0f\x0fPP\xff\xff\xff\x0f\x0f\x0f\xff\xff\xff\xff"
                    b"\xff\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f"
                    b"\x0f\x0f\x0f\x0f\xff\xff\xff\xff\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\xff\xff\xff\xff\xff\x0f\x0f\x0f\x0f\x0f\x0f\x0f"
                    b"\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0fO\x8f\x8f\x8f\x8f\x8f\x8f\x0f\x8f\xff\xff"
                    b"\xff\xff\xff\xff\x0f\x8f\xffPPPPP\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x8f\x8f\x8f\x8f\x8f\x8fO\x0f\xff\xff\xff\xff\xff\xff\x8f\x0fPPPPP\xff\x8f\x0f\x0f\x8f\xffP"
                    b"P\xbb\xbb\xbb\x0f\x8f\xffPP\xff\xff\xff\x0f\x8f\xffPP\xff\xbbP\x0f\x8f\xffPP\xff\xbbP\xbb\xbb\x88PP\xff\x8f\x0f\xff\xff\xbbPP\xff\x8f\x0fPPPPP\xff\x8f\x0fPPPP"
                    b"P\xff\x8f\x0f\x0f\x8f\xffPP\xff\xff\xff\x0f\x8f\xffPP\xff\xde\xbb\x0f\x8f\xffPP\xff\xde\xbb\x0f\x8f\xffPP\xff\xff\xff\xff\xbbPPP\xff\x8f\x0f\xbb\x88PPP\xff\x8f\x0f"
                    b"\xbb\xbb\x88PP\xff\x8f\x0f\xff\xff\xbbPP\xff\x8f\x0f\x0f\x8f\xffPPPPP\x0f\x8f\xff\xff\xff\xff\xff\xff\x0fO\x8f\x8f\x8f\x8f\x8f\x8f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f"
                    b"PPPPP\xff\x8f\x0f\xff\xff\xff\xff\xff\xff\x8f\x0f\x8f\x8f\x8f\x8f\x8f\x8fO\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x0f\x1fOOOOOO\x1fO\xde\xbb\xbb\xbb\xbb\xdeOO\xbb\x9a\xde"
                    b"\xdew\xbbOO\xbb\xbb\x88PP\xbbOO\xbb\xbb\xee\xbcP\xbbOO\xbb\xbb\xee\xdew\xbbOO\xde\xbb\xbb\xbb\xbb\xdeO\x1fOOOOOO\x1fo~~o\x00\x00\x00\x00~\xab\x88~\x00\x00\x00\x00"
                    b"~\xcd\x9a~\x00\x00\x00\x00o~~o\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                )
            ),
            PakFile(
                asset_id=564256465,
                asset_type="TXTR",
                should_compress=True,
                uncompressed_data=None,
                compressed_data=(
                    b"\x00\x00\x00\x00\x00 \x00 \x00\x00\x00\x04\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x02\x00\x00\x009\x00\x00\x04\xaf\x00\x01l\xff\x008\xef\xfe\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b'\x00\x00\x00\x00\x00\x00\x00\x00""#V\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02G\x025\x8b\xef\x8b\xdf\xff\xff'
                    b"\x00\x00\x00\x00\x00\x00\x00\x13\x00\x00\x02}\x00\x13|\xff$\x8c\xff\xff\xae\xff\xff\xc7\xff\xff\xc7 \xfe\xb6 \x00\x15\xbf\xff\xc6\x8e\xff\xfaA\xff\xfdr\x00"
                    b"\xff\xa4\x10\x00\xc7 \x00\x000\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xbb\xcc\xde\xff\xff\xff\xff\xff\xff\xfe\xed\xcaUU2!\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xeb\xff\xec\x97B\x86B\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x84 \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x14\x11$z\xdf\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x1a\x00\x00\x04\xdf"
                    b"\x00\x01\x8f\xe7\x01}\xfc@\x9d\xfdq\x00\xfdq\x00\x00\xee\xff\xff\xd9\xbb\xa8R\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00A\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00+\x00\x00\x08\xd8\x12j\xdb0\xdc\xb8 \x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x06\x00\x00\x00\x00\x16\x98\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff"
                )
            ),
            PakFile(
                asset_id=568030977,
                asset_type="TXTR",
                should_compress=True,
                uncompressed_data=None,
                compressed_data=(
                    b"\x00\x00\x00\x00\x00@\x00@\x00\x00\x00\x05\x00\x00\x00\x00\n\xaa\xaa\xaa\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xaa\n\xcc\xcc\xa0"
                    b"\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xa0\x00\x00"
                    b"\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xaa\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xa0\xcc\xcc\xcc\xa0"
                    b"\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xaa\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xaa\xaa\xa0"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\t\xcc\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff"
                    b"\x00\x00\x00\x00\xcc\xcc\xcc\xcc\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\xcc\xcc\xcc\xa8"
                    b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x04\xffP\x00\x00\x00\xfeP\x00\x00\xff\xf5\x00\x00"
                    b"\xff\xfe\x00\x00\xff\xff0\x00\xff\xffP\x00\xff\xff\x80\x00\xff\xff\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xaa\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x01h\x88\x88\xaf\xff\xff\xff\xff\xff\xff\xff\x00\x00\x04\xff\x00\x00\x04\xff\x00\x00\x04\xff\x00\x00\x04\xff\x00\x00\x04\xff\x88\x88\x8a\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff@\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x02\xff\x00\x00\t\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xfe\x88\x88\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff0\x00\xff\xfe\x00\x00\xff\xf4\x00\x00\xfe@\x00\x00@\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\xaa\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0e\xff\x00\x00\x0c\xff\x00\x00\x0b\xff"
                    b"\x00\x00\x04\xff\x00\x00\x00\xaf\x00\x00\x00\x08\xff\xfc\x00\x00\xff\xfdDD\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\x00\x00\x00\x00DDDD\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00D@\x00\x00"
                    b"\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\xaa\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x04x\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x88\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xaa\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc"
                    b"\n\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\xaa\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xaa"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\n\xcc\xcc\xa0"
                    b"\xaa\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xaa\xaa\xaa\xa0\x00\x00\x00\x00Wwww|\xcc\xcc\xcc|\xcc\xcc\xcc|\xc9ww|\xc7\x00\x00"
                    b"|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00u\x00\x00\x00\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00W\x00\x00\x00|\x00\x00\x00|\x00\x00\x00W\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00wwwu\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7ww\x9c\xc7"
                    b"\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7|\xc7\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00~\xee\xee\x00\x8f\xff\xff\x00\x8f\xff\xff\x00k\xbb\xbb\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05U\x00\x08\xff\xff\xee\xed\xb2\x00"
                    b"\xff\xff\xfe\x00\xff\xff\xff0\xbb\xcf\xffP\x00?\xffP\x00?\xffPU\x7f\xffP\xff\xff\xff@\x00\x00|\xc7\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00Wu\x00\x00"
                    b"|\xc7\x00\x00\x00O\xff\xff\x00\x8f\xff\xff\x00\x8f\xffU\x00\x8f\xfe\x00\x00\x8f\xfe3\x00\x9f\xff\xff\x00o\xff\xff\x00\n\xff\xff\xff\xff\xff\x10\xff\xff\xf8\x00"
                    b"UU \x00\x00\x00\x00\x00332\x00\xff\xff\xfb\x00\xff\xff\xfb\x00\xff\xff\xfb\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00Wu\x00\x00|\xc7|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc9ww|\xcc\xcc\xcc|\xcc\xcc\xccWwww\x00\x00%U\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00u\x00\x00\x00\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00UUT\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00W"
                    b"\x00\x00\x00|\x00\x00\x00|\x00\x00\x00W\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7ww\x9c\xc7\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7wwwu\x8a\xaa\x80\x00"
                    b"\xab\xaa\x80\x00\xaa\x00\x00\x00\xaa\x00\x00\x00\x88\x00\x0c\xff\x00\x00\x0b\xee\x00\x00\x00\x00\x00\x00\x03\xbc\x00\x08\xaa\xa8\x00\x08\xaa\xba\x00\x00\x00\xaa"
                    b"\x00\x00\x00\xaa\xff\xc0\x00\x88\xef\xf3\x00\x00\x0c\xf4\x00\x00\xce\xf3\x00\x00\x00\x00\x0c\xff\x00\x00\r\xf4\x00\x00\r\xfc\x88\x00\n\xff\xaa\x00\x00$"
                    b"\xaa\x00\x00\x00\xab\xaa\x80\x00\x8a\xaa\x80\x00\xff\xe0\x00\x00D\x10\x00\x00\xcc\xa0\x00\x00\xff\xe0\x00\x88D0\x00\xaa\x00\x00\x00\xaa\x00\x08\xaa\xba"
                    b"\x00\x08\xaa\xa8\xaa`\x06\xaa\xa0\x00\x00\n`\x8e\xfa\x06\x00\x18\xbc\x00\x00\x9e\xc7\x00`\x8f\xe8\x06\xa0\x02!\n\xaa`\x06\xaa\x82(\x00\x00\x00\x00)\xc2"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\x00"
                )
            ),
            PakFile(
                asset_id=638062812,
                asset_type="TXTR",
                should_compress=True,
                uncompressed_data=None,
                compressed_data=(
                    b"\x00\x00\x00\x00\x00@\x00@\x00\x00\x00\x05\x00\x00\x00\x00\n\xaa\xaa\xaa\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xaa\n\xcc\xcc\xa0"
                    b"\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xa0\x00\x00"
                    b"\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xaa\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xa0\xcc\xcc\xcc\xa0"
                    b"\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xaa\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xaa\xaa\xa0"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\t\xcc\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff"
                    b"\x00\x00\x00\x00\xcc\xcc\xcc\xcc\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\xcc\xcc\xcc\xa8"
                    b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x04\xffP\x00\x00\x00\xfeP\x00\x00\xff\xf5\x00\x00"
                    b"\xff\xfe\x00\x00\xff\xff0\x00\xff\xffP\x00\xff\xff\x80\x00\xff\xff\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xaa\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x01h\x88\x88\xaf\xff\xff\xff\xff\xff\xff\xff\x00\x00\x04\xff\x00\x00\x04\xff\x00\x00\x04\xff\x00\x00\x04\xff\x00\x00\x04\xff\x88\x88\x8a\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff\x80\x00\xff\xff@\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x02\xff\x00\x00\t\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0c\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xfe\x88\x88\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xfc\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff0\x00\xff\xfe\x00\x00\xff\xf4\x00\x00\xfe@\x00\x00@\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\xaa\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0c\xff\x00\x00\x0c\xff\x00\x00\x0e\xff\x00\x00\x0c\xff\x00\x00\x0b\xff"
                    b"\x00\x00\x04\xff\x00\x00\x00\xaf\x00\x00\x00\x08\xff\xfc\x00\x00\xff\xfdDD\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
                    b"\xff\xff\xff\xff\x00\x00\x00\x00DDDD\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00D@\x00\x00"
                    b"\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\xff\xf0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\xaa\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x04x\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x88\x88\x88\x88\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x88\x80\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0"
                    b"\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xa0\n\xcc\xcc\xaa\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc\n\xcc\xcc\xcc"
                    b"\n\xaa\xaa\xaa\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\xaa\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xcc\xa0\x00\x00\xaa\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\n\xaa\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xcc\x00\x00\n\xaa"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\xaa\xaa\xaa\xaa\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xcc\xaa\xaa\xaa\xaa\x00\x00\x00\x00\n\xcc\xcc\xa0"
                    b"\xaa\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xcc\xcc\xcc\xa0\xaa\xaa\xaa\xa0\x00\x00\x00\x00Wwww|\xcc\xcc\xcc|\xcc\xcc\xcc|\xc9ww|\xc7\x00\x00"
                    b"|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00u\x00\x00\x00\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00W\x00\x00\x00|\x00\x00\x00|\x00\x00\x00W\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00wwwu\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7ww\x9c\xc7"
                    b"\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7|\xc7\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00~\xee\xee\x00\x8f\xff\xff\x00\x8f\xff\xff\x00k\xbb\xbb\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x05U\x00\x08\xff\xff\xee\xed\xb2\x00"
                    b"\xff\xff\xfe\x00\xff\xff\xff0\xbb\xcf\xffP\x00?\xffP\x00?\xffPU\x7f\xffP\xff\xff\xff@\x00\x00|\xc7\x00\x00Wu\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00Wu\x00\x00"
                    b"|\xc7\x00\x00\x00O\xff\xff\x00\x8f\xff\xff\x00\x8f\xffU\x00\x8f\xfe\x00\x00\x8f\xfe3\x00\x9f\xff\xff\x00o\xff\xff\x00\n\xff\xff\xff\xff\xff\x10\xff\xff\xf8\x00"
                    b"UU \x00\x00\x00\x00\x00332\x00\xff\xff\xfb\x00\xff\xff\xfb\x00\xff\xff\xfb\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00Wu\x00\x00|\xc7|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc9ww|\xcc\xcc\xcc|\xcc\xcc\xccWwww\x00\x00%U\x00\x00\x00\x00"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00u\x00\x00\x00\xc7\x00\x00\x00\xc7\x00\x00\x00u\x00\x00\x00UUT\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00W"
                    b"\x00\x00\x00|\x00\x00\x00|\x00\x00\x00W\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7\x00\x00|\xc7ww\x9c\xc7\xcc\xcc\xcc\xc7\xcc\xcc\xcc\xc7wwwu\x8a\xaa\x80\x00"
                    b"\xab\xaa\x80\x00\xaa\x00\x00\x00\xaa\x00\x00\x00\x88\x00\x0c\xff\x00\x00\x0b\xee\x00\x00\x00\x00\x00\x00\x03\xbc\x00\x08\xaa\xa8\x00\x08\xaa\xba\x00\x00\x00\xaa"
                    b"\x00\x00\x00\xaa\xff\xc0\x00\x88\xef\xf3\x00\x00\x0c\xf4\x00\x00\xce\xf3\x00\x00\x00\x00\x0c\xff\x00\x00\r\xf4\x00\x00\r\xfc\x88\x00\n\xff\xaa\x00\x00$"
                    b"\xaa\x00\x00\x00\xab\xaa\x80\x00\x8a\xaa\x80\x00\xff\xe0\x00\x00D\x10\x00\x00\xcc\xa0\x00\x00\xff\xe0\x00\x88D0\x00\xaa\x00\x00\x00\xaa\x00\x08\xaa\xba"
                    b"\x00\x08\xaa\xa8\xaa`\x06\xaa\xa0\x00\x00\n`\x8e\xfa\x06\x00\x18\xbc\x00\x00\x9e\xc7\x00`\x8f\xe8\x06\xa0\x02!\n\xaa`\x06\xaa\x82(\x00\x00\x00\x00)\xc2"
                    b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\x00"
                )
            ),
        ]
    )

    result = PAK.build(source, target_game=game)
    decoded: PakBody = PAK.parse(result, target_game=game)

    for source_file in source.files:
        pad = 32 - len(source_file.compressed_data) % 32
        if pad < 32:
            source_file.compressed_data += b"\xFF" * pad

    assert decoded == source
