"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GeneratedPythonClientProject = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const generated_python_client_source_code_1 = require("./components/generated-python-client-source-code");
const open_api_generator_ignore_file_1 = require("./components/open-api-generator-ignore-file");
/**
 * Python project containing a python client (and lambda handler wrappers) generated using OpenAPI Generator CLI
 */
class GeneratedPythonClientProject extends python_1.PythonProject {
    constructor(options) {
        super({
            sample: false,
            pytest: false,
            ...options,
        });
        /**
         * The directory in which the built layer is output
         */
        this.layerDistDir = "dist/layer";
        // Store whether we've synthesized the project
        this.synthed = false;
        // Use a package.json to ensure the client is discoverable by nx
        new projen_1.JsonFile(this, "package.json", {
            obj: {
                name: this.name,
                __pdk__: true,
                version: options.version,
                scripts: Object.fromEntries(this.tasks.all.map((task) => [task.name, `npx projen ${task.name}`])),
            },
            readonly: true,
        });
        new open_api_generator_ignore_file_1.OpenApiGeneratorIgnoreFile(this);
        new generated_python_client_source_code_1.GeneratedPythonClientSourceCode(this, {
            specPath: options.specPath,
            invokeGenerator: options.generateClient,
        });
        // With pip and venv (default), it's useful to install our package into the shared venv to make
        // it easier for other packages in the monorepo to take dependencies on this package.
        if ((options.venv ?? true) && (options.pip ?? true)) {
            this.depsManager.installTask.exec("pip install --editable .");
        }
        // Package into a directory that can be used as a lambda layer. This is done as part of install since the end user
        // must control build order in the monorepo via explicit dependencies, and adding here means we can run as part of
        // initial project synthesis which ensures this is created regardless of whether the user has remembered to
        // configure build order.
        if (options.generateLayer) {
            const relativeLayerDir = path.join(".", this.layerDistDir, "python");
            this.depsManager.installTask.exec(`rm -rf ${relativeLayerDir}`);
            this.depsManager.installTask.exec(`pip install . --target ${relativeLayerDir}`);
        }
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.GeneratedPythonClientProject = GeneratedPythonClientProject;
//# sourceMappingURL=data:application/json;base64,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