"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tracer = void 0;
const commons_1 = require("@aws-lambda-powertools/commons");
const config_1 = require("./config");
const provider_1 = require("./provider");
const aws_xray_sdk_core_1 = require("aws-xray-sdk-core");
/**
 * ## Intro
 * Tracer is an opinionated thin wrapper for [AWS X-Ray SDK for Node.js](https://github.com/aws/aws-xray-sdk-node).
 *
 * Tracing data can be visualized through AWS X-Ray Console.
 *
 * ## Key features
 *   * Auto capture cold start as annotation, and responses or full exceptions as metadata
 *   * Auto-disable when not running in AWS Lambda environment
 *   * Automatically trace HTTP(s) clients and generate segments for each request
 *   * Support tracing functions via decorators, middleware, and manual instrumentation
 *   * Support tracing AWS SDK v2 and v3 via AWS X-Ray SDK for Node.js
 *
 * ## Usage
 *
 * For more usage examples, see [our documentation](https://awslabs.github.io/aws-lambda-powertools-typescript/latest/core/tracer/).
 *
 * ### Functions usage with middleware
 *
 * If you use function-based Lambda handlers you can use the [captureLambdaHandler()](./_aws_lambda_powertools_tracer.Tracer.html) middy middleware to automatically:
 * * handle the subsegment lifecycle
 * * add the `ServiceName` and `ColdStart` annotations
 * * add the function response as metadata
 * * add the function error as metadata (if any)
 *
 * @example
 * ```typescript
 * import { captureLambdaHandler, Tracer } from '@aws-lambda-powertools/tracer';
 * import middy from '@middy/core';
 *
 * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
 *
* const lambdaHandler = async (_event: any, _context: any) => {
 *   ...
 * };
 *
 * export const handler = middy(lambdaHandler).use(captureLambdaHandler(tracer));
 * ```
 *
 * ### Object oriented usage with decorators
 *
 * If instead you use TypeScript Classes to wrap your Lambda handler you can use the [@tracer.captureLambdaHandler()](./_aws_lambda_powertools_tracer.Tracer.html#captureLambdaHandler) decorator to automatically:
 * * handle the subsegment lifecycle
 * * add the `ServiceName` and `ColdStart` annotations
 * * add the function response as metadata
 * * add the function error as metadata (if any)
 *
 * @example
 * ```typescript
 * import { Tracer } from '@aws-lambda-powertools/tracer';
 * import { LambdaInterface } from '@aws-lambda-powertools/commons';
 *
 * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
 *
 * // FYI: Decorator might not render properly in VSCode mouse over due to https://github.com/microsoft/TypeScript/issues/47679 and might show as *@tracer* instead of `@tracer.captureLambdaHandler`
 *
 * class Lambda implements LambdaInterface {
 *   @tracer.captureLambdaHandler()
 *   public handler(event: any, context: any) {
 *     ...
 *   }
 * }
 *
 * const handlerClass = new Lambda();
 * export const handler = handlerClass.handler.bind(handlerClass);
 * ```
 *
 * ### Functions usage with manual instrumentation
 *
 * If you prefer to manually instrument your Lambda handler you can use the methods in the tracer class directly.
 *
 * @example
 * ```typescript
 * import { Tracer } from '@aws-lambda-powertools/tracer';
 *
 * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
 *
 * export const handler = async (_event: any, context: any) => {
 *   const segment = tracer.getSegment(); // This is the facade segment (the one that is created by AWS Lambda)
 *   // Create subsegment for the function & set it as active
 *   const subsegment = segment.addNewSubsegment(`## ${process.env._HANDLER}`);
 *   tracer.setSegment(subsegment);
 *
 *   // Annotate the subsegment with the cold start & serviceName
 *   tracer.annotateColdStart();
 *   tracer.addServiceNameAnnotation();
 *
 *   let res;
 *   try {
 *       // ... your own logic goes here
 *       // Add the response as metadata
 *       tracer.addResponseAsMetadata(res, process.env._HANDLER);
 *   } catch (err) {
 *       // Add the error as metadata
 *       tracer.addErrorAsMetadata(err as Error);
 *       throw err;
 *   } finally {
 *       // Close the subsegment
 *       subsegment.close();
 *       // Set the facade segment as active again
 *       tracer.setSegment(segment);
 *   }
 *
 *   return res;
 * }
 * ```
 */
class Tracer extends commons_1.Utility {
    constructor(options = {}) {
        super();
        this.captureError = true;
        this.captureHTTPsRequests = true;
        this.captureResponse = true;
        this.tracingEnabled = true;
        this.setOptions(options);
        this.provider = new provider_1.ProviderService();
        if (this.isTracingEnabled() && this.captureHTTPsRequests) {
            this.provider.captureHTTPsGlobal();
        }
        if (!this.isTracingEnabled()) {
            // Tell x-ray-sdk to not throw an error if context is missing but tracing is disabled
            this.provider.setContextMissingStrategy(() => ({}));
        }
    }
    /**
      * Add an error to the current segment or subsegment as metadata.
      *
      * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-errors
      *
      * @param error - Error to serialize as metadata
      */
    addErrorAsMetadata(error) {
        if (!this.isTracingEnabled()) {
            return;
        }
        const subsegment = this.getSegment();
        if (!this.captureError) {
            subsegment.addErrorFlag();
            return;
        }
        subsegment.addError(error, false);
    }
    /**
      * Add response data to the current segment or subsegment as metadata.
      *
      * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-annotations
      *
      * @param data - Data to serialize as metadata
      * @param methodName - Name of the method that is being traced
      */
    addResponseAsMetadata(data, methodName) {
        if (data === undefined || !this.captureResponse || !this.isTracingEnabled()) {
            return;
        }
        this.putMetadata(`${methodName} response`, data);
    }
    /**
     * Add service name to the current segment or subsegment as annotation.
     *
     */
    addServiceNameAnnotation() {
        if (!this.isTracingEnabled()) {
            return;
        }
        this.putAnnotation('Service', this.serviceName);
    }
    /**
     * Add ColdStart annotation to the current segment or subsegment.
     *
     * If Tracer has been initialized outside the Lambda handler then the same instance
     * of Tracer will be reused throughout the lifecycle of that same Lambda execution environment
     * and this method will annotate `ColdStart: false` after the first invocation.
     *
     * @see https://docs.aws.amazon.com/lambda/latest/dg/runtimes-context.html
     */
    annotateColdStart() {
        if (this.isTracingEnabled()) {
            this.putAnnotation('ColdStart', this.getColdStart());
        }
    }
    /**
     * Patch all AWS SDK v2 clients and create traces when your application makes calls to AWS services.
     *
     * If you want to patch a specific client use {@link captureAWSClient} and if you are using AWS SDK v3 use {@link captureAWSv3Client} instead.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-sdk-nodejs-awssdkclients.html
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     * const AWS = tracer.captureAWS(require('aws-sdk'));
     *
     * export const handler = async (_event: any, _context: any) => {
     *   ...
     * }
     * ```
     *
     * @param aws - AWS SDK v2 import
     * @returns AWS - Instrumented AWS SDK
     */
    captureAWS(aws) {
        if (!this.isTracingEnabled())
            return aws;
        return this.provider.captureAWS(aws);
    }
    /**
     * Patch a specific AWS SDK v2 client and create traces when your application makes calls to that AWS service.
     *
     * If you want to patch all clients use {@link captureAWS} and if you are using AWS SDK v3 use {@link captureAWSv3Client} instead.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-sdk-nodejs-awssdkclients.html
     *
     * @example
     * ```typescript
     * import { S3 } from 'aws-sdk';
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     * const s3 = tracer.captureAWSClient(new S3({ apiVersion: '2006-03-01' }));
     *
     * export const handler = async (_event: any, _context: any) => {
     *   ...
     * }
     * ```
     *
     * @param service - AWS SDK v2 client
     * @returns service - Instrumented AWS SDK v2 client
     */
    captureAWSClient(service) {
        if (!this.isTracingEnabled())
            return service;
        try {
            return this.provider.captureAWSClient(service);
        }
        catch (error) {
            try {
                // This is needed because some aws-sdk clients like AWS.DynamoDB.DocumentDB don't comply with the same
                // instrumentation contract like most base clients. 
                // For detailed explanation see: https://github.com/awslabs/aws-lambda-powertools-typescript/issues/524#issuecomment-1024493662
                this.provider.captureAWSClient(service.service);
                return service;
            }
            catch {
                throw error;
            }
        }
    }
    /**
     * Patch an AWS SDK v3 client and create traces when your application makes calls to that AWS service.
     *
     * If you are using AWS SDK v2 use {@link captureAWSClient} instead.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-sdk-nodejs-awssdkclients.html
     *
     * @example
     * ```typescript
     * import { S3Client } from '@aws-sdk/client-s3';
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     * const client = new S3Client({});
     * tracer.captureAWSv3Client(client);
     *
     * export const handler = async (_event: any, _context: any) => {
     *   ...
     * }
     * ```
     *
     * @param service - AWS SDK v3 client
     * @returns service - Instrumented AWS SDK v3 client
     */
    captureAWSv3Client(service) {
        if (!this.isTracingEnabled())
            return service;
        return this.provider.captureAWSv3Client(service);
    }
    /**
     * A decorator automating capture of metadata and annotations on segments or subsegments for a Lambda Handler.
     *
     * Using this decorator on your handler function will automatically:
     * * handle the subsegment lifecycle
     * * add the `ColdStart` annotation
     * * add the function response as metadata
     * * add the function error as metadata (if any)
     *
     * Note: Currently TypeScript only supports decorators on classes and methods. If you are using the
     * function syntax, you should use the middleware instead.
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     * import { LambdaInterface } from '@aws-lambda-powertools/commons';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     *
     * class Lambda implements LambdaInterface {
     *   @tracer.captureLambdaHandler()
     *   public handler(event: any, context: any) {
     *     ...
     *   }
     * }
     *
     * const handlerClass = new Lambda();
     * export const handler = handlerClass.handler.bind(handlerClass);
     * ```
     *
     * @decorator Class
     * @param options - (_optional_) Options for the decorator
     */
    captureLambdaHandler(options) {
        return (_target, _propertyKey, descriptor) => {
            /**
             * The descriptor.value is the method this decorator decorates, it cannot be undefined.
             */
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const originalMethod = descriptor.value;
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            const tracerRef = this;
            // Use a function() {} instead of an () => {} arrow function so that we can
            // access `myClass` as `this` in a decorated `myClass.myMethod()`.
            descriptor.value = (function (event, context, callback) {
                // eslint-disable-next-line @typescript-eslint/no-this-alias
                const handlerRef = this;
                if (!tracerRef.isTracingEnabled()) {
                    return originalMethod.apply(handlerRef, [event, context, callback]);
                }
                return tracerRef.provider.captureAsyncFunc(`## ${process.env._HANDLER}`, async (subsegment) => {
                    var _a;
                    tracerRef.annotateColdStart();
                    tracerRef.addServiceNameAnnotation();
                    let result;
                    try {
                        result = await originalMethod.apply(handlerRef, [event, context, callback]);
                        if ((_a = options === null || options === void 0 ? void 0 : options.captureResponse) !== null && _a !== void 0 ? _a : true) {
                            tracerRef.addResponseAsMetadata(result, process.env._HANDLER);
                        }
                    }
                    catch (error) {
                        tracerRef.addErrorAsMetadata(error);
                        throw error;
                    }
                    finally {
                        subsegment === null || subsegment === void 0 ? void 0 : subsegment.close();
                        subsegment === null || subsegment === void 0 ? void 0 : subsegment.flush();
                    }
                    return result;
                });
            });
            return descriptor;
        };
    }
    /**
     * A decorator automating capture of metadata and annotations on segments or subsegments for an arbitrary function.
     *
     * Using this decorator on your function will automatically:
     * * handle the subsegment lifecycle
     * * add the function response as metadata
     * * add the function error as metadata (if any)
     *
     * Note: Currently TypeScript only supports decorators on classes and methods. If you are using the
     * function syntax, you should use the middleware instead.
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     * import { LambdaInterface } from '@aws-lambda-powertools/commons';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     *
     * class Lambda implements LambdaInterface {
     *   @tracer.captureMethod()
     *   public myMethod(param: any) {
     *     ...
     *   }
     *
     *   public handler(event: any, context: any) {
     *     ...
     *   }
     * }
     *
     * const handlerClass = new Lambda();
     * export const handler = handlerClass.handler.bind(handlerClass);;
     * ```
     *
     * @decorator Class
     * @param options - (_optional_) Options for the decorator
     */
    captureMethod(options) {
        return (_target, propertyKey, descriptor) => {
            // The descriptor.value is the method this decorator decorates, it cannot be undefined.
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const originalMethod = descriptor.value;
            // eslint-disable-next-line @typescript-eslint/no-this-alias
            const tracerRef = this;
            // Use a function() {} instead of an () => {} arrow function so that we can
            // access `myClass` as `this` in a decorated `myClass.myMethod()`.
            descriptor.value = function (...args) {
                if (!tracerRef.isTracingEnabled()) {
                    return originalMethod.apply(this, [...args]);
                }
                const methodName = String(propertyKey);
                const subsegmentName = (options === null || options === void 0 ? void 0 : options.subSegmentName) ? options.subSegmentName : `### ${methodName}`;
                return tracerRef.provider.captureAsyncFunc(subsegmentName, async (subsegment) => {
                    var _a;
                    let result;
                    try {
                        result = await originalMethod.apply(this, [...args]);
                        if ((_a = options === null || options === void 0 ? void 0 : options.captureResponse) !== null && _a !== void 0 ? _a : true) {
                            tracerRef.addResponseAsMetadata(result, methodName);
                        }
                    }
                    catch (error) {
                        tracerRef.addErrorAsMetadata(error);
                        throw error;
                    }
                    finally {
                        subsegment === null || subsegment === void 0 ? void 0 : subsegment.close();
                        subsegment === null || subsegment === void 0 ? void 0 : subsegment.flush();
                    }
                    return result;
                });
            };
            return descriptor;
        };
    }
    /**
     * Get the current root AWS X-Ray trace id.
     *
     * Utility method that returns the current AWS X-Ray Root trace id. Useful as correlation id for downstream processes.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-traces
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     *
     * export const handler = async () => {
     *   try {
     *     ...
     *   } catch (err) {
     *     const rootTraceId = tracer.getRootXrayTraceId();
     *
     *     // Example of returning an error response
     *     return {
     *       statusCode: 500,
     *       // Include the rootTraceId in the response so we can show a "contact support" button that
     *       // takes the customer to a customer service form with the trace as additional context.
     *       body: `Internal Error - Please contact support and quote the following id: ${rootTraceId}`,
     *       headers: { '_X_AMZN_TRACE_ID': rootTraceId },
     *     };
     *   }
     * }
     * ```
     *
     * @returns string - The root X-Ray trace id.
     */
    getRootXrayTraceId() {
        return this.envVarsService.getXrayTraceId();
    }
    /**
     * Get the active segment or subsegment in the current scope.
     *
     * Usually you won't need to call this method unless you are creating custom subsegments or using manual mode.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-segments
     * @see https://awslabs.github.io/aws-lambda-powertools-typescript/latest/core/tracer/#escape-hatch-mechanism
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     *
     * export const handler = async (_event: any, _context: any) => {
     *   const currentSegment = tracer.getSegment();
     *   ... // Do something with segment
     * }
     * ```
     *
     * @returns segment - The active segment or subsegment in the current scope.
     */
    getSegment() {
        if (!this.isTracingEnabled()) {
            return new aws_xray_sdk_core_1.Subsegment('## Dummy segment');
        }
        const segment = this.provider.getSegment();
        if (segment === undefined) {
            throw new Error('Failed to get the current sub/segment from the context.');
        }
        return segment;
    }
    /**
     * Get the current value of the `tracingEnabled` property.
     *
     * You can use this method during manual instrumentation to determine
     * if tracer is currently enabled.
     *
     * @returns tracingEnabled - `true` if tracing is enabled, `false` otherwise.
     */
    isTracingEnabled() {
        return this.tracingEnabled;
    }
    /**
     * Adds annotation to existing segment or subsegment.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-sdk-nodejs-segment.html#xray-sdk-nodejs-segment-annotations
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     *
     * export const handler = async (_event: any, _context: any) => {
     *   tracer.putAnnotation('successfulBooking', true);
     * }
     * ```
     *
     * @param key - Annotation key
     * @param value - Value for annotation
     */
    putAnnotation(key, value) {
        if (!this.isTracingEnabled())
            return;
        const document = this.getSegment();
        if (document instanceof aws_xray_sdk_core_1.Segment) {
            console.warn('You cannot annotate the main segment in a Lambda execution environment');
            return;
        }
        document.addAnnotation(key, value);
    }
    /**
     * Adds metadata to existing segment or subsegment.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-sdk-nodejs-segment.html#xray-sdk-nodejs-segment-metadata
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     *
     * export const handler = async (_event: any, _context: any) => {
     *   const res = someLogic();
     *   tracer.putMetadata('paymentResponse', res);
     * }
     * ```
     *
     * @param key - Metadata key
     * @param value - Value for metadata
     * @param namespace - Namespace that metadata will lie under, if none is passed it will use the serviceName
     */
    putMetadata(key, value, namespace) {
        if (!this.isTracingEnabled())
            return;
        const document = this.getSegment();
        if (document instanceof aws_xray_sdk_core_1.Segment) {
            console.warn('You cannot add metadata to the main segment in a Lambda execution environment');
            return;
        }
        namespace = namespace || this.serviceName;
        document.addMetadata(key, value, namespace);
    }
    /**
     * Sets the passed subsegment as the current active subsegment.
     *
     * If you are using a middleware or a decorator this is done automatically for you.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-sdk-nodejs-subsegments.html
     *
     * @example
     * ```typescript
     * import { Tracer } from '@aws-lambda-powertools/tracer';
     * import { Subsegment } from 'aws-xray-sdk-core';
     *
     * const tracer = new Tracer({ serviceName: 'serverlessAirline' });
     *
     * export const handler = async (_event: any, _context: any) => {
     *   const subsegment = new Subsegment('### foo.bar');
     *   tracer.setSegment(subsegment);
     * }
     * ```
     *
     * @param segment - Subsegment to set as the current segment
     */
    setSegment(segment) {
        if (!this.isTracingEnabled())
            return;
        return this.provider.setSegment(segment);
    }
    /**
     * Getter for `customConfigService`.
     * Used internally during initialization.
     */
    getCustomConfigService() {
        return this.customConfigService;
    }
    /**
     * Getter for `envVarsService`.
     * Used internally during initialization.
     */
    getEnvVarsService() {
        return this.envVarsService;
    }
    /**
     * Determine if we are running inside an Amplify CLI process.
     * Used internally during initialization.
     */
    isAmplifyCli() {
        return this.getEnvVarsService().getAwsExecutionEnv() === 'AWS_Lambda_amplify-mock';
    }
    /**
     * Determine if we are running in a Lambda execution environment.
     * Used internally during initialization.
     */
    isLambdaExecutionEnv() {
        return this.getEnvVarsService().getAwsExecutionEnv() !== '';
    }
    /**
     * Determine if we are running inside a SAM CLI process.
     * Used internally during initialization.
     */
    isLambdaSamCli() {
        return this.getEnvVarsService().getSamLocal() !== '';
    }
    /**
     * Setter for `captureError` based on configuration passed and environment variables.
     * Used internally during initialization.
     */
    setCaptureError() {
        var _a;
        const customConfigValue = (_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getTracingCaptureError();
        if (customConfigValue !== undefined && customConfigValue.toLowerCase() === 'false') {
            this.captureError = false;
            return;
        }
        const envVarsValue = this.getEnvVarsService().getTracingCaptureError();
        if (envVarsValue.toLowerCase() === 'false') {
            this.captureError = false;
            return;
        }
    }
    /**
     * Patch all HTTP(s) clients and create traces when your application makes calls outgoing calls.
     *
     * Calls using third-party HTTP request libraries, such as Axios, are supported as long as they use the native http
     * module under the hood. Support for third-party HTTP request libraries is provided on a best effort basis.
     *
     * @see https://docs.aws.amazon.com/xray/latest/devguide/xray-sdk-nodejs-httpclients.html
     *
     * @param enabled - Whether or not to patch all HTTP clients
     * @returns void
     */
    setCaptureHTTPsRequests(enabled) {
        var _a;
        if (enabled !== undefined && !enabled) {
            this.captureHTTPsRequests = false;
            return;
        }
        const customConfigValue = (_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getCaptureHTTPsRequests();
        if (customConfigValue !== undefined && customConfigValue.toLowerCase() === 'false') {
            this.captureHTTPsRequests = false;
            return;
        }
        const envVarsValue = this.getEnvVarsService().getCaptureHTTPsRequests();
        if (envVarsValue.toLowerCase() === 'false') {
            this.captureHTTPsRequests = false;
            return;
        }
    }
    /**
     * Setter for `captureResponse` based on configuration passed and environment variables.
     * Used internally during initialization.
     */
    setCaptureResponse() {
        var _a;
        const customConfigValue = (_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getTracingCaptureResponse();
        if (customConfigValue !== undefined && customConfigValue.toLowerCase() === 'false') {
            this.captureResponse = false;
            return;
        }
        const envVarsValue = this.getEnvVarsService().getTracingCaptureResponse();
        if (envVarsValue.toLowerCase() === 'false') {
            this.captureResponse = false;
            return;
        }
    }
    /**
     * Setter for `customConfigService` based on configuration passed.
     * Used internally during initialization.
     *
     * @param customConfigService - Custom configuration service to use
     */
    setCustomConfigService(customConfigService) {
        this.customConfigService = customConfigService ? customConfigService : undefined;
    }
    /**
     * Setter and initializer for `envVarsService`.
     * Used internally during initialization.
     */
    setEnvVarsService() {
        this.envVarsService = new config_1.EnvironmentVariablesService();
    }
    /**
     * Method that reconciles the configuration passed with the environment variables.
     * Used internally during initialization.
     *
     * @param options - Configuration passed to the tracer
     */
    setOptions(options) {
        const { enabled, serviceName, captureHTTPsRequests, customConfigService } = options;
        this.setEnvVarsService();
        this.setCustomConfigService(customConfigService);
        this.setTracingEnabled(enabled);
        this.setCaptureResponse();
        this.setCaptureError();
        this.setServiceName(serviceName);
        this.setCaptureHTTPsRequests(captureHTTPsRequests);
        return this;
    }
    /**
     * Setter for `customConfigService` based on configurations passed and environment variables.
     * Used internally during initialization.
     *
     * @param serviceName - Name of the service to use
     */
    setServiceName(serviceName) {
        var _a;
        if (serviceName !== undefined && this.isValidServiceName(serviceName)) {
            this.serviceName = serviceName;
            return;
        }
        const customConfigValue = (_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getServiceName();
        if (customConfigValue !== undefined && this.isValidServiceName(customConfigValue)) {
            this.serviceName = customConfigValue;
            return;
        }
        const envVarsValue = this.getEnvVarsService().getServiceName();
        if (envVarsValue !== undefined && this.isValidServiceName(envVarsValue)) {
            this.serviceName = envVarsValue;
            return;
        }
        this.serviceName = this.getDefaultServiceName();
    }
    /**
     * Setter for `tracingEnabled` based on configurations passed and environment variables.
     * Used internally during initialization.
     *
     * @param enabled - Whether or not tracing is enabled
     */
    setTracingEnabled(enabled) {
        var _a;
        if (enabled !== undefined && !enabled) {
            this.tracingEnabled = enabled;
            return;
        }
        const customConfigValue = (_a = this.getCustomConfigService()) === null || _a === void 0 ? void 0 : _a.getTracingEnabled();
        if (customConfigValue !== undefined && customConfigValue.toLowerCase() === 'false') {
            this.tracingEnabled = false;
            return;
        }
        const envVarsValue = this.getEnvVarsService().getTracingEnabled();
        if (envVarsValue.toLowerCase() === 'false') {
            this.tracingEnabled = false;
            return;
        }
        if (this.isAmplifyCli() || this.isLambdaSamCli() || !this.isLambdaExecutionEnv()) {
            this.tracingEnabled = false;
        }
    }
}
exports.Tracer = Tracer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiVHJhY2VyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL1RyYWNlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSw0REFBb0Y7QUFFcEYscUNBQStFO0FBRS9FLHlDQUF1RTtBQUN2RSx5REFBd0Q7QUFFeEQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0EwR0c7QUFDSCxNQUFNLE1BQU8sU0FBUSxpQkFBTztJQW9CMUIsWUFBbUIsVUFBeUIsRUFBRTtRQUM1QyxLQUFLLEVBQUUsQ0FBQztRQWpCRixpQkFBWSxHQUFZLElBQUksQ0FBQztRQUU3Qix5QkFBb0IsR0FBWSxJQUFJLENBQUM7UUFFckMsb0JBQWUsR0FBWSxJQUFJLENBQUM7UUFVaEMsbUJBQWMsR0FBWSxJQUFJLENBQUM7UUFLckMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUN6QixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksMEJBQWUsRUFBRSxDQUFDO1FBQ3RDLElBQUksSUFBSSxDQUFDLGdCQUFnQixFQUFFLElBQUksSUFBSSxDQUFDLG9CQUFvQixFQUFFO1lBQ3hELElBQUksQ0FBQyxRQUFRLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztTQUNwQztRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsRUFBRTtZQUM1QixxRkFBcUY7WUFDckYsSUFBSSxDQUFDLFFBQVEsQ0FBQyx5QkFBeUIsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7U0FDckQ7SUFDSCxDQUFDO0lBRUQ7Ozs7OztRQU1JO0lBQ0csa0JBQWtCLENBQUMsS0FBWTtRQUNwQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLEVBQUU7WUFDNUIsT0FBTztTQUNSO1FBRUQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3JDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ3RCLFVBQVUsQ0FBQyxZQUFZLEVBQUUsQ0FBQztZQUUxQixPQUFPO1NBQ1I7UUFFRCxVQUFVLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7UUFPSTtJQUNHLHFCQUFxQixDQUFDLElBQWMsRUFBRSxVQUFtQjtRQUM5RCxJQUFJLElBQUksS0FBSyxTQUFTLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLEVBQUU7WUFDM0UsT0FBTztTQUNSO1FBRUQsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLFVBQVUsV0FBVyxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7O09BR0c7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFO1lBQzVCLE9BQU87U0FDUjtRQUNELElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsRUFBRTtZQUMzQixJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztTQUN0RDtJQUNILENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BcUJHO0lBQ0ksVUFBVSxDQUFJLEdBQU07UUFDekIsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRTtZQUFFLE9BQU8sR0FBRyxDQUFDO1FBRXpDLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bc0JHO0lBQ0ksZ0JBQWdCLENBQUksT0FBVTtRQUNuQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQUUsT0FBTyxPQUFPLENBQUM7UUFFN0MsSUFBSTtZQUNGLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztTQUNoRDtRQUFDLE9BQU8sS0FBSyxFQUFFO1lBQ2QsSUFBSTtnQkFDRixzR0FBc0c7Z0JBQ3RHLG9EQUFvRDtnQkFDcEQsK0hBQStIO2dCQUMvSCxJQUFJLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFFLE9BQThCLENBQUMsT0FBTyxDQUFDLENBQUM7Z0JBRXhFLE9BQU8sT0FBTyxDQUFDO2FBQ2hCO1lBQUMsTUFBTTtnQkFDTixNQUFNLEtBQUssQ0FBQzthQUNiO1NBQ0Y7SUFDSCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BdUJHO0lBQ0ksa0JBQWtCLENBQUksT0FBVTtRQUNyQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQUUsT0FBTyxPQUFPLENBQUM7UUFFN0MsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQ0c7SUFDSSxvQkFBb0IsQ0FBQyxPQUFxQztRQUMvRCxPQUFPLENBQUMsT0FBTyxFQUFFLFlBQVksRUFBRSxVQUFVLEVBQUUsRUFBRTtZQUMzQzs7ZUFFRztZQUNILG9FQUFvRTtZQUNwRSxNQUFNLGNBQWMsR0FBRyxVQUFVLENBQUMsS0FBTSxDQUFDO1lBRXpDLDREQUE0RDtZQUM1RCxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUM7WUFDdkIsMkVBQTJFO1lBQzNFLGtFQUFrRTtZQUNsRSxVQUFVLENBQUMsS0FBSyxHQUFHLENBQUMsVUFBeUIsS0FBSyxFQUFFLE9BQU8sRUFBRSxRQUFRO2dCQUNuRSw0REFBNEQ7Z0JBQzVELE1BQU0sVUFBVSxHQUFZLElBQUksQ0FBQztnQkFFakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFO29CQUNqQyxPQUFPLGNBQWMsQ0FBQyxLQUFLLENBQUMsVUFBVSxFQUFFLENBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxRQUFRLENBQUUsQ0FBQyxDQUFDO2lCQUN2RTtnQkFFRCxPQUFPLFNBQVMsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxPQUFPLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxFQUFFLEtBQUssRUFBQyxVQUFVLEVBQUMsRUFBRTs7b0JBQzFGLFNBQVMsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO29CQUM5QixTQUFTLENBQUMsd0JBQXdCLEVBQUUsQ0FBQztvQkFDckMsSUFBSSxNQUFlLENBQUM7b0JBQ3BCLElBQUk7d0JBQ0YsTUFBTSxHQUFHLE1BQU0sY0FBYyxDQUFDLEtBQUssQ0FBQyxVQUFVLEVBQUUsQ0FBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLFFBQVEsQ0FBRSxDQUFDLENBQUM7d0JBQzlFLElBQUksTUFBQSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsZUFBZSxtQ0FBSSxJQUFJLEVBQUU7NEJBQ3BDLFNBQVMsQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQzt5QkFDL0Q7cUJBRUY7b0JBQUMsT0FBTyxLQUFLLEVBQUU7d0JBQ2QsU0FBUyxDQUFDLGtCQUFrQixDQUFDLEtBQWMsQ0FBQyxDQUFDO3dCQUM3QyxNQUFNLEtBQUssQ0FBQztxQkFDYjs0QkFBUzt3QkFDUixVQUFVLGFBQVYsVUFBVSx1QkFBVixVQUFVLENBQUUsS0FBSyxFQUFFLENBQUM7d0JBQ3BCLFVBQVUsYUFBVixVQUFVLHVCQUFWLFVBQVUsQ0FBRSxLQUFLLEVBQUUsQ0FBQztxQkFDckI7b0JBRUQsT0FBTyxNQUFNLENBQUM7Z0JBQ2hCLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFpRCxDQUFDO1lBRW5ELE9BQU8sVUFBVSxDQUFDO1FBQ3BCLENBQUMsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FtQ0c7SUFDSSxhQUFhLENBQUMsT0FBOEI7UUFDakQsT0FBTyxDQUFDLE9BQU8sRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLEVBQUU7WUFDMUMsdUZBQXVGO1lBQ3ZGLG9FQUFvRTtZQUNwRSxNQUFNLGNBQWMsR0FBRyxVQUFVLENBQUMsS0FBTSxDQUFDO1lBRXpDLDREQUE0RDtZQUM1RCxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUM7WUFDdkIsMkVBQTJFO1lBQzNFLGtFQUFrRTtZQUNsRSxVQUFVLENBQUMsS0FBSyxHQUFHLFVBQVUsR0FBRyxJQUFlO2dCQUM3QyxJQUFJLENBQUMsU0FBUyxDQUFDLGdCQUFnQixFQUFFLEVBQUU7b0JBQ2pDLE9BQU8sY0FBYyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDLENBQUM7aUJBQzlDO2dCQUVELE1BQU0sVUFBVSxHQUFHLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDdkMsTUFBTSxjQUFjLEdBQUcsQ0FBQSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsY0FBYyxFQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxPQUFPLFVBQVUsRUFBRSxDQUFDO2dCQUU5RixPQUFPLFNBQVMsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxFQUFFLEtBQUssRUFBQyxVQUFVLEVBQUMsRUFBRTs7b0JBQzVFLElBQUksTUFBTSxDQUFDO29CQUNYLElBQUk7d0JBQ0YsTUFBTSxHQUFHLE1BQU0sY0FBYyxDQUFDLEtBQUssQ0FBQyxJQUFJLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDLENBQUM7d0JBQ3JELElBQUksTUFBQSxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsZUFBZSxtQ0FBSSxJQUFJLEVBQUU7NEJBQ3BDLFNBQVMsQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7eUJBQ3JEO3FCQUNGO29CQUFDLE9BQU8sS0FBSyxFQUFFO3dCQUNkLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFjLENBQUMsQ0FBQzt3QkFFN0MsTUFBTSxLQUFLLENBQUM7cUJBQ2I7NEJBQVM7d0JBQ1IsVUFBVSxhQUFWLFVBQVUsdUJBQVYsVUFBVSxDQUFFLEtBQUssRUFBRSxDQUFDO3dCQUNwQixVQUFVLGFBQVYsVUFBVSx1QkFBVixVQUFVLENBQUUsS0FBSyxFQUFFLENBQUM7cUJBQ3JCO29CQUVELE9BQU8sTUFBTSxDQUFDO2dCQUNoQixDQUFDLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQztZQUVGLE9BQU8sVUFBVSxDQUFDO1FBQ3BCLENBQUMsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQ0c7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDLGNBQWMsRUFBRSxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BcUJHO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsRUFBRTtZQUM1QixPQUFPLElBQUksOEJBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1NBQzNDO1FBQ0QsTUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUMzQyxJQUFJLE9BQU8sS0FBSyxTQUFTLEVBQUU7WUFDekIsTUFBTSxJQUFJLEtBQUssQ0FBQyx5REFBeUQsQ0FBQyxDQUFDO1NBQzVFO1FBRUQsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVEOzs7Ozs7O09BT0c7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDO0lBQzdCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksYUFBYSxDQUFDLEdBQVcsRUFBRSxLQUFnQztRQUNoRSxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFO1lBQUUsT0FBTztRQUVyQyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDbkMsSUFBSSxRQUFRLFlBQVksMkJBQU8sRUFBRTtZQUMvQixPQUFPLENBQUMsSUFBSSxDQUFDLHdFQUF3RSxDQUFDLENBQUM7WUFFdkYsT0FBTztTQUNSO1FBQ0QsUUFBUSxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQW9CRztJQUNJLFdBQVcsQ0FBQyxHQUFXLEVBQUUsS0FBYyxFQUFFLFNBQThCO1FBQzVFLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFBRSxPQUFPO1FBRXJDLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUNuQyxJQUFJLFFBQVEsWUFBWSwyQkFBTyxFQUFFO1lBQy9CLE9BQU8sQ0FBQyxJQUFJLENBQUMsK0VBQStFLENBQUMsQ0FBQztZQUU5RixPQUFPO1NBQ1I7UUFFRCxTQUFTLEdBQUcsU0FBUyxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUM7UUFDMUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BcUJHO0lBQ0ksVUFBVSxDQUFDLE9BQTZCO1FBQzdDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFBRSxPQUFPO1FBRXJDLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7T0FHRztJQUNLLHNCQUFzQjtRQUM1QixPQUFPLElBQUksQ0FBQyxtQkFBbUIsQ0FBQztJQUNsQyxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssaUJBQWlCO1FBQ3ZCLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQztJQUM3QixDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssWUFBWTtRQUNsQixPQUFPLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLGtCQUFrQixFQUFFLEtBQUsseUJBQXlCLENBQUM7SUFDckYsQ0FBQztJQUVEOzs7T0FHRztJQUNLLG9CQUFvQjtRQUMxQixPQUFPLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLGtCQUFrQixFQUFFLEtBQUssRUFBRSxDQUFDO0lBQzlELENBQUM7SUFFRDs7O09BR0c7SUFDSyxjQUFjO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUMsV0FBVyxFQUFFLEtBQUssRUFBRSxDQUFDO0lBQ3ZELENBQUM7SUFFRDs7O09BR0c7SUFDSyxlQUFlOztRQUNyQixNQUFNLGlCQUFpQixHQUFHLE1BQUEsSUFBSSxDQUFDLHNCQUFzQixFQUFFLDBDQUFFLHNCQUFzQixFQUFFLENBQUM7UUFDbEYsSUFBSSxpQkFBaUIsS0FBSyxTQUFTLElBQUksaUJBQWlCLENBQUMsV0FBVyxFQUFFLEtBQUssT0FBTyxFQUFFO1lBQ2xGLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDO1lBRTFCLE9BQU87U0FDUjtRQUVELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLHNCQUFzQixFQUFFLENBQUM7UUFDdkUsSUFBSSxZQUFZLENBQUMsV0FBVyxFQUFFLEtBQUssT0FBTyxFQUFFO1lBQzFDLElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDO1lBRTFCLE9BQU87U0FDUjtJQUNILENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ssdUJBQXVCLENBQUMsT0FBaUI7O1FBQy9DLElBQUksT0FBTyxLQUFLLFNBQVMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNyQyxJQUFJLENBQUMsb0JBQW9CLEdBQUcsS0FBSyxDQUFDO1lBRWxDLE9BQU87U0FDUjtRQUVELE1BQU0saUJBQWlCLEdBQUcsTUFBQSxJQUFJLENBQUMsc0JBQXNCLEVBQUUsMENBQUUsdUJBQXVCLEVBQUUsQ0FBQztRQUNuRixJQUFJLGlCQUFpQixLQUFLLFNBQVMsSUFBSSxpQkFBaUIsQ0FBQyxXQUFXLEVBQUUsS0FBSyxPQUFPLEVBQUU7WUFDbEYsSUFBSSxDQUFDLG9CQUFvQixHQUFHLEtBQUssQ0FBQztZQUVsQyxPQUFPO1NBQ1I7UUFFRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyx1QkFBdUIsRUFBRSxDQUFDO1FBQ3hFLElBQUksWUFBWSxDQUFDLFdBQVcsRUFBRSxLQUFLLE9BQU8sRUFBRTtZQUMxQyxJQUFJLENBQUMsb0JBQW9CLEdBQUcsS0FBSyxDQUFDO1lBRWxDLE9BQU87U0FDUjtJQUNILENBQUM7SUFFRDs7O09BR0c7SUFDSyxrQkFBa0I7O1FBQ3hCLE1BQU0saUJBQWlCLEdBQUcsTUFBQSxJQUFJLENBQUMsc0JBQXNCLEVBQUUsMENBQUUseUJBQXlCLEVBQUUsQ0FBQztRQUNyRixJQUFJLGlCQUFpQixLQUFLLFNBQVMsSUFBSSxpQkFBaUIsQ0FBQyxXQUFXLEVBQUUsS0FBSyxPQUFPLEVBQUU7WUFDbEYsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUM7WUFFN0IsT0FBTztTQUNSO1FBRUQsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUMseUJBQXlCLEVBQUUsQ0FBQztRQUMxRSxJQUFJLFlBQVksQ0FBQyxXQUFXLEVBQUUsS0FBSyxPQUFPLEVBQUU7WUFDMUMsSUFBSSxDQUFDLGVBQWUsR0FBRyxLQUFLLENBQUM7WUFFN0IsT0FBTztTQUNSO0lBQ0gsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssc0JBQXNCLENBQUMsbUJBQTRDO1FBQ3pFLElBQUksQ0FBQyxtQkFBbUIsR0FBRyxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztJQUNuRixDQUFDO0lBRUQ7OztPQUdHO0lBQ0ssaUJBQWlCO1FBQ3ZCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxvQ0FBMkIsRUFBRSxDQUFDO0lBQzFELENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLFVBQVUsQ0FBQyxPQUFzQjtRQUN2QyxNQUFNLEVBQ0osT0FBTyxFQUNQLFdBQVcsRUFDWCxvQkFBb0IsRUFDcEIsbUJBQW1CLEVBQ3BCLEdBQUcsT0FBTyxDQUFDO1FBRVosSUFBSSxDQUFDLGlCQUFpQixFQUFFLENBQUM7UUFDekIsSUFBSSxDQUFDLHNCQUFzQixDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBQzFCLElBQUksQ0FBQyxlQUFlLEVBQUUsQ0FBQztRQUN2QixJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQ2pDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBRW5ELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssY0FBYyxDQUFDLFdBQW9COztRQUN6QyxJQUFJLFdBQVcsS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQ3JFLElBQUksQ0FBQyxXQUFXLEdBQUcsV0FBVyxDQUFDO1lBRS9CLE9BQU87U0FDUjtRQUVELE1BQU0saUJBQWlCLEdBQUcsTUFBQSxJQUFJLENBQUMsc0JBQXNCLEVBQUUsMENBQUUsY0FBYyxFQUFFLENBQUM7UUFDMUUsSUFBSSxpQkFBaUIsS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLGlCQUFpQixDQUFDLEVBQUU7WUFDakYsSUFBSSxDQUFDLFdBQVcsR0FBRyxpQkFBaUIsQ0FBQztZQUVyQyxPQUFPO1NBQ1I7UUFFRCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxjQUFjLEVBQUUsQ0FBQztRQUMvRCxJQUFJLFlBQVksS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLFlBQVksQ0FBQyxFQUFFO1lBQ3ZFLElBQUksQ0FBQyxXQUFXLEdBQUcsWUFBWSxDQUFDO1lBRWhDLE9BQU87U0FDUjtRQUNELElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLHFCQUFxQixFQUFFLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ssaUJBQWlCLENBQUMsT0FBaUI7O1FBQ3pDLElBQUksT0FBTyxLQUFLLFNBQVMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNyQyxJQUFJLENBQUMsY0FBYyxHQUFHLE9BQU8sQ0FBQztZQUU5QixPQUFPO1NBQ1I7UUFFRCxNQUFNLGlCQUFpQixHQUFHLE1BQUEsSUFBSSxDQUFDLHNCQUFzQixFQUFFLDBDQUFFLGlCQUFpQixFQUFFLENBQUM7UUFDN0UsSUFBSSxpQkFBaUIsS0FBSyxTQUFTLElBQUksaUJBQWlCLENBQUMsV0FBVyxFQUFFLEtBQUssT0FBTyxFQUFFO1lBQ2xGLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDO1lBRTVCLE9BQU87U0FDUjtRQUVELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLGlCQUFpQixFQUFFLENBQUM7UUFDbEUsSUFBSSxZQUFZLENBQUMsV0FBVyxFQUFFLEtBQUssT0FBTyxFQUFFO1lBQzFDLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDO1lBRTVCLE9BQU87U0FDUjtRQUVELElBQUksSUFBSSxDQUFDLFlBQVksRUFBRSxJQUFJLElBQUksQ0FBQyxjQUFjLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsRUFBRSxFQUFFO1lBQ2hGLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDO1NBQzdCO0lBQ0gsQ0FBQztDQUVGO0FBR0Msd0JBQU0ifQ==