"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPythonSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample python source code
 */
exports.getPythonSampleSource = (options) => ({
    "__init__.py": "#",
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.py": `from dataclasses import fields
from ${options.openApiGatewayPackageName} import OpenApiGatewayLambdaApi, OpenApiIntegration
from ${options.pythonClientPackageName}.api.default_api_operation_config import OperationLookup, OperationConfig
from ${options.moduleName} import SPEC

class Api(OpenApiGatewayLambdaApi):
    """
    Type-safe construct for the API Gateway resources defined by the spec.
    You will likely not need to modify this file, and can instead extend it and define your integrations.
    """
    def __init__(self, scope, id, integrations: OperationConfig[OpenApiIntegration], **kwargs):
        super().__init__(scope, id,
            **kwargs,
            integrations={ field.name: getattr(integrations, field.name) for field in fields(integrations) },
            spec=SPEC,
            operation_lookup=OperationLookup,
        )
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample_api.py": `from ${options.openApiGatewayPackageName} import Authorizers, OpenApiIntegration
from ${options.pythonClientPackageName}.api.default_api_operation_config import OperationConfig
from aws_cdk.aws_lambda import LayerVersion, Code, Function, Runtime
from .api import Api
from constructs import Construct
from ${options.moduleName} import get_generated_client_layer_directory
from pathlib import Path
from os import path

class SampleApi(Construct):
  """
  An example of how to add integrations to your api
  """
  def __init__(self, scope, id):
      super().__init__(scope, id)

      # Layer which contains the generated client.
      self.generated_client_layer = LayerVersion(self, 'GeneratedClientLayer',
          code=Code.from_asset(get_generated_client_layer_directory())
      )

      self.api = Api(self, 'Api',
          default_authorizer=Authorizers.iam(),
          integrations=OperationConfig(
              say_hello=OpenApiIntegration(
                  function=Function(self, 'SayHello',
                      runtime=Runtime.PYTHON_3_9,
                      code=Code.from_asset(path.join(str(Path(__file__).parent.absolute()), 'handlers')),
                      handler="say_hello_handler_sample.handler",
                      layers=[self.generated_client_layer],
                  ),
              ),
          ),
      )
`,
            // Generate an example lambda handler
            "handlers/say_hello_handler_sample.py": `from ${options.pythonClientPackageName}.api.default_api_operation_config import say_hello_handler, SayHelloRequest, ApiResponse
from ${options.pythonClientPackageName}.model.api_error import ApiError
from ${options.pythonClientPackageName}.model.hello_response import HelloResponse

@say_hello_handler
def handler(input: SayHelloRequest, **kwargs) -> ApiResponse[HelloResponse, ApiError]:
    """
    An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs
    """
    return ApiResponse(
        status_code=200,
        body=HelloResponse(message="Hello {}!".format(input.request_parameters["name"])),
        headers={}
    )
`,
        }
        : {}),
});
//# sourceMappingURL=data:application/json;base64,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