"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const ddb = require("@aws-cdk/aws-dynamodb");
const api = require("@aws-cdk/aws-apigateway");
test("check properties", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {};
    const construct = new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb-default", apiGatewayToDynamoDBProps);
    expect(construct.dynamoTable !== null);
    expect(construct.apiGateway !== null);
    expect(construct.apiGatewayRole !== null);
    expect(construct.apiGatewayCloudWatchRole !== null);
    expect(construct.apiGatewayLogGroup !== null);
});
test("check allow CRUD operations", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowReadOperation: true,
        allowCreateOperation: true,
        createRequestTemplate: "{}",
        allowDeleteOperation: true,
        allowUpdateOperation: true,
        updateRequestTemplate: "{}",
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:PutItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:DeleteItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
            ],
            Version: "2012-10-17",
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4",
            },
        ],
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "PUT",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "DELETE",
        AuthorizationType: "AWS_IAM",
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{id}",
    });
});
test("check allow read and update only", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        allowUpdateOperation: true,
        updateRequestTemplate: "{}",
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "dynamodb:Query",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
                {
                    Action: "dynamodb:UpdateItem",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": ["testapigatewaydynamodbDynamoTableEEE3F463", "Arn"],
                    },
                },
            ],
            Version: "2012-10-17",
        },
        PolicyName: "testapigatewaydynamodbapigatewayroleDefaultPolicy43AC565D",
        Roles: [
            {
                Ref: "testapigatewaydynamodbapigatewayrole961B19C4",
            },
        ],
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "AWS_IAM",
    });
});
test("check using custom partition key for dynamodb", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        dynamoTableProps: {
            partitionKey: {
                name: "page_id",
                type: ddb.AttributeType.STRING,
            },
        },
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: "{page_id}",
    });
});
test("override apiGatewayProps for api gateway", () => {
    const stack = new core_1.Stack();
    const apiGatewayToDynamoDBProps = {
        apiGatewayProps: {
            description: "This is a sample description for api gateway",
        },
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb", apiGatewayToDynamoDBProps);
    expect(stack).toHaveResource("AWS::ApiGateway::RestApi", {
        Description: "This is a sample description for api gateway",
    });
});
test("Test deployment ApiGateway AuthorizationType override", () => {
    const stack = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack, "api-gateway-dynamodb", {
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE,
            },
        },
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "GET",
        AuthorizationType: "NONE",
    });
});
test("Test deployment with existing DynamoDB table", () => {
    const oddPartitionKeyName = 'oddName';
    const oddReadCapacity = 23;
    const stack = new core_1.Stack();
    const table = new ddb.Table(stack, "existing-table", {
        partitionKey: {
            name: oddPartitionKeyName,
            type: ddb.AttributeType.STRING,
        },
        readCapacity: oddReadCapacity
    });
    const apiGatewayToDynamoDBProps = {
        existingTableObj: table
    };
    new lib_1.ApiGatewayToDynamoDB(stack, "test-api-gateway-dynamodb-default", apiGatewayToDynamoDBProps);
    // Confirm there is only the one table
    expect(stack).toCountResources("AWS::DynamoDB::Table", 1);
    // Confirm that the one table is the one create here
    expect(stack).toHaveResourceLike("AWS::DynamoDB::Table", {
        ProvisionedThroughput: {
            ReadCapacityUnits: oddReadCapacity,
        }
    });
    expect(stack).toHaveResource("AWS::ApiGateway::Resource", {
        PathPart: `{${oddPartitionKeyName}}`,
    });
});
test("check setting allowReadOperation=false for dynamodb", () => {
    const stack1 = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack1, "test-api-gateway-dynamodb1", {
        allowReadOperation: true,
        allowDeleteOperation: true
    });
    // Expect two APIG Methods (GET, DELETE) for allowReadOperation and allowDeleteOperation
    expect(stack1).toCountResources("AWS::ApiGateway::Method", 2);
    const stack2 = new core_1.Stack();
    new lib_1.ApiGatewayToDynamoDB(stack2, "test-api-gateway-dynamodb2", {
        allowReadOperation: false,
        allowDeleteOperation: true
    });
    // Expect only one APIG Method (DELETE) for allowDeleteOperation
    expect(stack2).toCountResources("AWS::ApiGateway::Method", 1);
});
//# sourceMappingURL=data:application/json;base64,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