"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToDynamoDB = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
/**
 * @summary The ApiGatewayToDynamoDB class.
 */
class ApiGatewayToDynamoDB extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToDynamoDB class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CloudFrontToApiGatewayToLambdaProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        let partitionKeyName;
        let dynamoTableProps;
        // Set the default props for DynamoDB table
        if (props.dynamoTableProps) {
            dynamoTableProps = core_2.overrideProps(defaults.DefaultTableProps, props.dynamoTableProps);
            partitionKeyName = dynamoTableProps.partitionKey.name;
        }
        else {
            partitionKeyName = defaults.DefaultTableProps.partitionKey.name;
            dynamoTableProps = defaults.DefaultTableProps;
        }
        if (props.existingTableObj) {
            partitionKeyName = core_2.getPartitionKeyNameFromTable(props.existingTableObj);
        }
        this.dynamoTable = defaults.buildDynamoDBTable(this, {
            existingTableObj: props.existingTableObj,
            dynamoTableProps,
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway Resource
        const apiGatewayResource = this.apiGateway.root.addResource("{" + partitionKeyName + "}");
        // Setup API Gateway Method
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = props.createRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:PutItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "PutItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "POST",
                apiResource: this.apiGateway.root,
                requestTemplate: createRequestTemplate
            });
        }
        // Read
        if (props.allowReadOperation === undefined || props.allowReadOperation === true) {
            let readRequestTemplate;
            if (props.readRequestTemplate) {
                readRequestTemplate = props.readRequestTemplate;
            }
            else {
                readRequestTemplate =
                    `{ \
          "TableName": "${this.dynamoTable.tableName}", \
          "KeyConditionExpression": "${partitionKeyName} = :v1", \
          "ExpressionAttributeValues": { \
            ":v1": { \
              "S": "$input.params('${partitionKeyName}')" \
            } \
          } \
        }`;
            }
            this.addActionToPolicy("dynamodb:Query");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "Query",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "GET",
                apiResource: apiGatewayResource,
                requestTemplate: readRequestTemplate
            });
        }
        // Update
        if (props.allowUpdateOperation && props.allowUpdateOperation === true && props.updateRequestTemplate) {
            const updateRequestTemplate = props.updateRequestTemplate.replace("${Table}", this.dynamoTable.tableName);
            this.addActionToPolicy("dynamodb:UpdateItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "UpdateItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "PUT",
                apiResource: apiGatewayResource,
                requestTemplate: updateRequestTemplate
            });
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            let deleteRequestTemplate;
            if (props.deleteRequestTemplate) {
                deleteRequestTemplate = props.deleteRequestTemplate;
            }
            else {
                deleteRequestTemplate =
                    `{ \
          "TableName": "${this.dynamoTable.tableName}", \
          "Key": { \
            "${partitionKeyName}": { \
              "S": "$input.params('${partitionKeyName}')" \
              } \
            }, \
          "ReturnValues": "ALL_OLD" \
        }`;
            }
            this.addActionToPolicy("dynamodb:DeleteItem");
            defaults.addProxyMethodToApiResource({
                service: "dynamodb",
                action: "DeleteItem",
                apiGatewayRole: this.apiGatewayRole,
                apiMethod: "DELETE",
                apiResource: apiGatewayResource,
                requestTemplate: deleteRequestTemplate
            });
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.dynamoTable.tableArn
            ],
            actions: [`${action}`]
        }));
    }
}
exports.ApiGatewayToDynamoDB = ApiGatewayToDynamoDB;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToDynamoDB[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-dynamodb.ApiGatewayToDynamoDB", version: "1.139.0" };
//# sourceMappingURL=data:application/json;base64,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